use std::error::Error;

use bevy::log::{
    tracing_subscriber::{
        self,
        filter::{FromEnvError, ParseError},
        prelude::*,
        reload, EnvFilter, Registry,
    },
    BoxedLayer, Level, DEFAULT_FILTER,
};

pub struct LoggingConf {
    pub filter: String,
    pub level: Option<Level>,
}
impl Default for LoggingConf {
    fn default() -> Self {
        Self {
            filter: DEFAULT_FILTER.to_string(),
            level: Some(Level::INFO),
        }
    }
}
impl LoggingConf {
    pub fn create_filter(&self) -> EnvFilter {
        // taken from bevy LogPlugin
        let default_filter = {
            format!(
                "{},{}",
                self.level.map(|l| l.as_str()).unwrap_or("off"),
                self.filter
            )
        };
        EnvFilter::try_from_default_env()
            .or_else(|from_env_error| {
                _ = from_env_error
                    .source()
                    .and_then(|source| source.downcast_ref::<ParseError>())
                    .map(|parse_err| {
                        // we cannot use the `error!` macro here because the logger is not ready yet.
                        eprintln!("LogPlugin failed to parse filter from env: {}", parse_err);
                    });
                Ok::<EnvFilter, FromEnvError>(EnvFilter::builder().parse_lossy(&default_filter))
            })
            .unwrap()
    }
}

pub struct LoggingController {
    handle: reload::Handle<EnvFilter, tracing_subscriber::layer::Layered<BoxedLayer, Registry>>,
}
impl LoggingController {
    /// initializes the tracing subscriber, that logs to stdout
    /// returns a handle which can be used to replace the filter
    /// Consists of two filters
    #[cfg(not(target_os = "android"))]
    pub fn init(initial_conf: LoggingConf) -> Self {
        Self::init_raw(initial_conf, || {
            #[cfg(not(any(target_arch = "wasm32", target_os = "android")))]
            let layer = tracing_subscriber::fmt::Layer::default().with_writer(std::io::stderr);

            #[cfg(target_arch = "wasm32")]
            let layer = tracing_wasm::WASMLayer::default();

            Box::new(layer)
        })
    }
    pub fn init_raw(initial_conf: LoggingConf, layer_init: fn() -> BoxedLayer) -> Self {
        let registry = tracing_subscriber::registry();

        let layer = layer_init();

        // assume info as default level for now
        // should be reloaded to represent the user-set log level
        let filter = initial_conf.create_filter();
        let (filter, reload_handle) = reload::Layer::new(filter);

        let registry = registry.with(layer).with(filter);

        // setup file logger
        #[cfg(all(
            feature = "file-logger",
            not(any(target_arch = "wasm32", target_arch = "wasm64"))
        ))]
        let registry = registry.with(tracing_subscriber::fmt::layer().with_writer({
            // read log output directory from TTE_LOG_DIR environment variable
            // if unset use the application data directory
            // in case this is unavailable use the current directory
            let data_dir = std::env::var("TTE_LOG_DIR")
                .unwrap_or(crate::settings::get_data_dir().unwrap_or(String::from("./")));
            println!("Using {} as log directory", data_dir);
            // create a file appender
            // this can be used for write operations and supports log rotataion
            tracing_appender::rolling::hourly(data_dir, format!("{}.log", crate::APP_ID))
        }));

        if registry.try_init().is_err() {
            // XXX: We should be able to use error! here,
            // because if the global logger cannot be set,
            // it must have already been set, meaning that a logger
            // has already been configured
            bevy::log::error!("Unable to set global logger");
        }

        Self {
            handle: reload_handle,
        }
    }
    pub fn update(&self, conf: LoggingConf) -> Result<(), reload::Error> {
        self.handle.modify(|filter| *filter = conf.create_filter())
    }
}
