#[cfg(feature = "graphics")]
mod loader;

#[cfg(feature = "graphics")]
pub mod locales;

use bevy::prelude::*;

#[cfg(feature = "graphics")]
use fluent::{bundle::FluentBundle, FluentResource};

#[cfg(feature = "graphics")]
use unic_langid::LanguageIdentifier;

#[cfg(feature = "graphics")]
use self::{
    loader::{FluentResourceAsset, FluentResourceLoader},
    locales::LocaleHandles,
};

#[cfg(feature = "graphics")]
use crate::{prelude::*, settings::ActiveSettingsBank, AppState};

#[derive(Default)]
#[cfg(feature = "graphics")]
pub struct TranslationsPlugin;

#[cfg(feature = "graphics")]
impl Plugin for TranslationsPlugin {
    fn build(&self, app: &mut App) {
        app.init_asset_loader::<FluentResourceLoader>();
        app.init_asset::<FluentResourceAsset>();
        app.init_resource::<Localization>();
        app.init_resource::<LocaleHandles>();
        // load language set in config
        #[cfg(feature = "graphics")]
        app.add_systems(OnExit(AppState::LoadApp), select_language);
        // automatically change language
        #[cfg(feature = "graphics")]
        app.add_systems(
            Update,
            select_language.run_if(resource_changed::<ActiveSettingsBank>),
        );
        app.register_asset_collection::<LocaleHandles>();
    }
}

/// Resource used to translate text
#[derive(Default, Resource)]
pub struct Localization(
    #[cfg(feature = "graphics")]
    Option<FluentBundle<FluentResource, intl_memoizer::concurrent::IntlLangMemoizer>>,
);

#[cfg(feature = "graphics")]
impl Localization {
    fn new() -> Self {
        Self(None)
    }
    fn with_lang(
        mut self,
        lang: LanguageIdentifier,
        handles: &LocaleHandles,
        assets: &Assets<FluentResourceAsset>,
    ) -> Self {
        let mut lang_formats = Vec::with_capacity(3);
        let mut lang_resources = Vec::with_capacity(3);

        // add the application default language to the stack
        {
            let (handle, lang) = handles.get_default_lang();
            if let Some(handle) = assets.get(&handle) {
                lang_formats.push(lang);
                lang_resources.push(handle);
            }
        }

        // add the default language for the language-region group to the stack
        // this is the users choosen locale in most cases
        if let Ok(lang) = lang.language.as_str().parse::<LanguageIdentifier>() {
            if let Some(handle) = handles.get_lang(&lang).and_then(|h| assets.get(&h)) {
                lang_formats.push(lang.clone());
                lang_resources.push(handle);
            }
        }

        // add the specific language to the language stack
        // most of the time this is unset, but allows using overlay locales
        // like de_CH
        if let Some(Some(handle)) = handles.get_lang(&lang).map(|h| assets.get(&h)) {
            lang_formats.push(lang.clone());
            lang_resources.push(handle);
        }

        let mut bundle = FluentBundle::new_concurrent(lang_formats);
        for res in lang_resources {
            // FluentResource doesn't implement the copy trait
            // and I don't want to move it from Assets,
            // as it is still needed when changing the languages later
            let source = res.0.source();
            // we can expect this to succeed, but better be safe than sorry
            if let Ok(res) = FluentResource::try_new(source.to_string()) {
                bundle.add_resource_overriding(res);
            }
        }

        self.0 = Some(bundle);
        self
    }
}

/// change locale if the language to the language used in the settings menu
#[cfg(feature = "graphics")]
fn select_language(
    settings: Res<ActiveSettingsBank>,
    mut localization: ResMut<Localization>,
    handles: Res<LocaleHandles>,
    assets: Res<Assets<FluentResourceAsset>>,
) {
    *localization = Localization::new().with_lang(
        settings.lang.get_lang_id(),
        handles.as_ref(),
        assets.as_ref(),
    );
}

pub trait Translate {
    /// Try and translate the id
    /// if a translation message was found, it will be returned
    /// otherwise None will be returned
    fn try_translate(&self, id: &str) -> Option<String>;
    /// Get the translation unit by the id
    /// returns the translation ID if no message was found
    fn translate(&self, id: &str) -> String {
        self.try_translate(id).unwrap_or(String::from(id))
    }
    /// Attempts to get the translation unit by ID
    /// if it was successful, it returns the translated message.
    /// If there is no translation unit with the ID,
    /// the fallback message will be returned.
    /// shorthand for: `.translate(id).unwrap_or(fallback.to_string())`
    fn translate_fallback(&self, id: &str, fallback: &str) -> String {
        self.try_translate(id).unwrap_or(String::from(fallback))
    }
}
impl Translate for Localization {
    #[cfg(feature = "graphics")]
    fn try_translate(&self, id: &str) -> Option<String> {
        if let Some(bundle) = &self.0 {
            if let Some(Some(pat)) = bundle.get_message(id).map(|msg| msg.value()) {
                let mut errors = Vec::new();
                let translation = bundle.format_pattern(pat, None, &mut errors);
                if errors.is_empty() {
                    return Some(translation.to_string());
                }
            }
        }
        None
    }
    #[cfg(not(feature = "graphics"))]
    fn try_translate(&self, _id: &str) -> Option<String> {
        None
    }
}
