use super::{Localization, Translate};
use crate::languages;
use serde::{Deserialize, Serialize};
use sys_locale::get_locale;
use unic_langid::{langid, LanguageIdentifier};

languages![
    // List of available languages
    // the first entry in the list will be used as a default fallback language
    (en, En, "en", "English"),
    // new languages have to be added here
    // format:
    // (struct_variant, EnumVariant, File_Variant, "Native Language Name"),
    (de, De, "de", "Deutsch"),
    (ru, Ru, "ru", "русский язык"),
    (pt_br, PtBR, "pt_BR", "português do Brasil"),
    (fr, Fr, "fr", "français"),
];

#[macro_export]
/// automatically creates an enum containg all available languages (and Auto)
/// as well as interop-bindings and an asset loading resource
/// the first entry in the list is the default fallback language
macro_rules! languages {
    (($rstruct_id: ident, $renum_id:ident, $rfile_id:literal, $rname: literal), $(($struct_id: ident, $enum_id:ident, $file_id:literal, $name:literal),)*) => {
        #[derive(Serialize, Deserialize, Default, Debug, Clone, PartialEq)]
        pub enum Language {
            #[serde(rename = "auto")]
            #[default]
            /// Automatically determine the Language (JIT)
            /// by checking the system language
            Auto,
            /// the default fallback locale used by the app
            #[doc = $rname]
            $renum_id,
            $(
                #[serde(rename = $file_id)]
                #[doc = $name]
                $enum_id,
            )*
        }
        impl Language {
            /// returns a list of all Languages,
            /// except Auto
            pub fn list() -> Vec<Self> {
                vec![Self::$renum_id, $(Self::$enum_id,)*]
            }

            /// returns the display name of the language
            /// in the specify language
            pub fn get_name(&self, localization: &Localization) -> String {
                match self {
                    Self::$renum_id => $rname.to_string(),
                    Self::Auto => localization
                        .translate("config-language-system_language")
                        .to_string(),
                    $(Self::$enum_id => $name.to_string(),)*
                }
            }
            pub fn get_lang_id(&self) -> LanguageIdentifier {
                match self {
                    Self::Auto => {
                        if let Some(id) = get_locale() {
                            if let Ok(id) = id.parse() {
                                return id;
                            }
                        }
                        Language::default_fallback().get_lang_id()
                    },
                    Self::$renum_id => langid!($rfile_id),
                    $(Self::$enum_id => langid!($file_id),)*
                }
            }
            /// returns the closest match to the language identifier
            /// from the list of supported languages
            /// returns the default language, if no match can be found
            pub fn from_lang_id(lang: &LanguageIdentifier) -> Self {
                let id = lang.region.map(|r|format!("{}_{}", lang.language.as_str(), r.as_str())).unwrap_or(lang.language.as_str().to_string());
                match id.as_str() {
                    $rfile_id => Self::$renum_id,
                    $($file_id => Self::$enum_id,)*
                    _ => Language::default_fallback(),
                }
            }
            /// resolves the language if auto,
            /// will always return one of the supported languages
            pub fn resolve(&self) -> Self {
                 match self {
                     Self::Auto => {
                         if let Some(id) = get_locale() {
                             if let Ok(id) = id.parse() {
                                 return Language::from_lang_id(&id);
                             }
                         }
                         Language::default_fallback()
                     },
                     Self::$renum_id => Self::$renum_id,
                     $(Self::$enum_id => Self::$enum_id,)*
                 }
            }
            /// return language id for use in the online docs
            pub fn get_book_id(&self) -> &'static str {
                match self.resolve() {
                    Language::De => "de",
                    Language::En => "en",
                    _ => "en"
                }
            }
            /// get the application default fallback language
            /// this will be used if no other languages are available
            pub fn default_fallback() -> Self {
                Self::$renum_id
            }
        }

        use bevy::prelude::*;
        use super::loader::FluentResourceAsset;
        #[derive(Resource, Default)]
        pub struct LocaleHandles {
            /// Handle to the required default fallback language
            /// this will always be loaded
            /// guaranteed to be loaded after the AppLoad state
            #[doc = $rname]
            pub $rstruct_id: Handle<FluentResourceAsset>,
            $(
                /// Handle to the language
                /// if this is None, the language is not available
                /// if the asset loaded successfully, the option is set to Some(Handle)
                /// if the asset failed to load, the option has the variant None
                /// During the AppLoad phase the option will always be set to the Some variant
                #[doc = $name]
                pub $struct_id: Option<Handle<FluentResourceAsset>>,
            )*
        }
        impl LocaleHandles {
            /// get the handle to the asset by the language id
            pub fn get_lang(&self, lang: &LanguageIdentifier) -> Option<Handle<FluentResourceAsset>> {
                let id = lang.region.map(|r|format!("{}_{}", lang.language.as_str(), r.as_str())).unwrap_or(lang.language.as_str().to_string());
                match id.as_str() {
                    $rfile_id => Some(self.$rstruct_id.clone_weak()),
                    $($file_id => self.$struct_id.as_ref().map(|handle| handle.clone_weak()),)*
                    _ => None,
                }
            }
            /// return a handle to the default language
            pub fn get_default_lang(&self) -> (Handle<FluentResourceAsset>, LanguageIdentifier) {
                (self.$rstruct_id.clone_weak(), langid!($rfile_id))
            }
       }
            use $crate::errors::ErrorDisplay;
        impl $crate::game::asset_loading::LoadableAssetCollection for LocaleHandles {
            fn load_all(asset_server: &AssetServer) -> Self {
                Self {
                    $rstruct_id: asset_server.load(concat!("locales/", $rfile_id, "/translations.ftl")),
                    $(
                    $struct_id: Some(asset_server.load(concat!("locales/", $file_id, "/translations.ftl"))),
                    )*
                }
           }
            fn check_all(&mut self, asset_server: &AssetServer) -> $crate::game::asset_loading::AssetLoadStateDump {
                let mut dump = $crate::game::asset_loading::AssetLoadStateDump::default();

                dump.requested += 1;
                // only the default language has to exist
                // it doesn't really matter if the others are missing
                match asset_server.get_load_state(&self.$rstruct_id) {
                    Some(bevy::asset::LoadState::Loaded) => dump.loaded += 1,
                    Some(bevy::asset::LoadState::Failed(_)) => dump.failed += 1,
                    _ => ()
                }

                // check optional assets
                $(
                    dump.requested += 1;
                    if let Some(handle) = &self.$struct_id {
                        match asset_server.get_load_state(handle) {
                            Some(bevy::asset::LoadState::Loaded) => dump.loaded += 1,
                            Some(bevy::asset::LoadState::Failed(_)) => {
                                dump.ignored += 1;
                                // mark asset as unavailable
                                self.$struct_id = None;
                            },
                            _ => ()
                        }
                    } else {
                        // optional asset failed to load in an earlier frame
                        dump.ignored += 1;
                    }
                )*

                dump
            }
            fn get_error(&self, localization: Option<&$crate::i18n::Localization>) -> Option<ErrorDisplay> {
                // only the en_US language fails, the other locales are ignored
                // this means that if this module had one failed asset,
                // it has to have been en_US
                Some(ErrorDisplay {
                    title: localization
                        .and_then(|l| l.try_translate("missing-default-language-title"))
                        .unwrap_or(String::from("Failed to load default Language")),
                    description: localization
                        .and_then(|l| l.try_translate("missing-default-language-description"))
                        .unwrap_or(String::from("Running the game without it wouldn't make sense, as it is used as a fallback for missing translations.")),
                    link: None,
                })
            }
        }
    };
}
