use std::time::Duration;

use bevy::{
    asset::Assets,
    pbr::{NotShadowCaster, NotShadowReceiver, StandardMaterial},
    prelude::*,
    render::{
        mesh::{Mesh, PrimitiveTopology},
        render_asset::RenderAssetUsages,
    },
    transform::components::Transform,
    utils::HashMap,
};

use crate::{
    game::controls::PanOrbitCam, prelude::on_timer_real, settings::ActiveSettingsBank,
    ui::DebugMaterial, AppState,
};

use super::cell_water::{start_cell_water, CellWater};

/// amount mesh is repeated
const WATER_MESH_RADIUS: i32 = 10;
const WATER_TOTAL_MESH_RADIUS: isize = 8;

/// size cell_water rect
const WATER_MESH_SIZE: f32 = 5.0;
pub const WATER_HEIGHT: f32 = 0.15;
pub const GROUND_HEIGHT: f32 = -0.1;

const WATER_TOTAL_SIZE: f32 = (WATER_MESH_RADIUS as f32 * 2.0 + 1.0) * WATER_MESH_SIZE;

pub struct WaterPlugin;
impl Plugin for WaterPlugin {
    fn build(&self, app: &mut App) {
        app.add_systems(
            OnEnter(AppState::InGame),
            spawn_water.after(start_cell_water),
        );
        app.add_systems(
            Update,
            spawn_water
                .run_if(on_timer_real(Duration::from_secs(1)))
                .run_if(in_state(AppState::InGame)),
        );
        app.add_systems(
            Update,
            (despawn_water::<false>, spawn_water.after(start_cell_water))
                .run_if(resource_changed::<ActiveSettingsBank>)
                .run_if(in_state(AppState::InGame)),
        );
        app.add_systems(OnExit(AppState::InGame), despawn_water::<true>);
        app.init_resource::<WaterData>();
    }
}

#[derive(Resource, Default)]
pub struct WaterData {
    map: HashMap<(isize, isize), Entity>,
    mesh: Option<Handle<Mesh>>,
    material: Option<Handle<StandardMaterial>>,
}

#[derive(Component)]
pub struct Water;

pub fn spawn_water(
    mut commands: Commands,
    mut materials: ResMut<Assets<StandardMaterial>>,
    mut meshes: ResMut<Assets<Mesh>>,
    cell_water: Res<CellWater>,
    settings: Res<ActiveSettingsBank>,
    mut data: ResMut<WaterData>,
    cam: Query<&PanOrbitCam>,
) {
    if !settings.graphics.water {
        return;
    }

    let Ok(cam) = cam.get_single() else {
        return;
    };

    let Some(ref water) = cell_water.image_handle else {
        error!("Started water, but cell water texture was not ready!");
        return;
    };

    let mesh = data
        .mesh
        .get_or_insert_with(|| {
            info!("Start rendering water texture");
            let mut positions: Vec<[f32; 3]> = Vec::new();
            let mut uv: Vec<[f32; 2]> = Vec::new();

            for x in -WATER_MESH_RADIUS..=WATER_MESH_RADIUS {
                for y in -WATER_MESH_RADIUS..=WATER_MESH_RADIUS {
                    let x_c = (x as f32 - 0.5) * WATER_MESH_SIZE;
                    let y_c = (y as f32 - 0.5) * WATER_MESH_SIZE;

                    positions.push([x_c, 0.0, y_c]);
                    positions.push([x_c, 0.0, y_c + WATER_MESH_SIZE]);
                    positions.push([x_c + WATER_MESH_SIZE, 0.0, y_c]);
                    positions.push([x_c + WATER_MESH_SIZE, 0.0, y_c]);
                    positions.push([x_c, 0.0, y_c + WATER_MESH_SIZE]);
                    positions.push([x_c + WATER_MESH_SIZE, 0.0, y_c + WATER_MESH_SIZE]);

                    uv.push([0.0, 1.0]);
                    uv.push([1.0, 1.0]);
                    uv.push([0.0, 0.0]);
                    uv.push([0.0, 0.0]);
                    uv.push([1.0, 1.0]);
                    uv.push([1.0, 0.0]);
                }
            }

            let normals: Vec<[f32; 3]> = vec![[0., 1., 0.]; positions.len()];
            let mesh = Mesh::new(
                PrimitiveTopology::TriangleList,
                RenderAssetUsages::default(),
            )
            .with_inserted_attribute(Mesh::ATTRIBUTE_POSITION, positions)
            .with_inserted_attribute(Mesh::ATTRIBUTE_UV_0, uv)
            .with_inserted_attribute(Mesh::ATTRIBUTE_NORMAL, normals)
            .with_generated_tangents()
            .unwrap();
            meshes.add(mesh)
        })
        .clone();

    let material = data
        .material
        .get_or_insert_with(|| {
            materials.add(StandardMaterial {
                base_color: Color::WHITE,
                base_color_texture: Some(water.clone()),
                alpha_mode: AlphaMode::Blend,
                //unlit: true,
                ..default()
            })
        })
        .clone();

    let center = cam.focus / WATER_TOTAL_SIZE;
    let center_x = center.x.round() as isize;
    let center_z = center.z.round() as isize;

    data.map.retain(|pos, entity| {
        let d_x = pos.0 - center_x;
        let d_z = pos.1 - center_z;
        if d_x * d_x + d_z * d_z > WATER_TOTAL_MESH_RADIUS * WATER_TOTAL_MESH_RADIUS {
            commands.entity(*entity).despawn_recursive();
            false
        } else {
            true
        }
    });

    for x in -WATER_TOTAL_MESH_RADIUS..=WATER_TOTAL_MESH_RADIUS {
        for z in -WATER_TOTAL_MESH_RADIUS..=WATER_TOTAL_MESH_RADIUS {
            if x * x + z * z <= WATER_TOTAL_MESH_RADIUS * WATER_TOTAL_MESH_RADIUS {
                let x = x + center_x;
                let z = z + center_z;
                if !data.map.contains_key(&(x, z)) {
                    let entity = commands
                        .spawn((
                            Mesh3d(mesh.clone()),
                            MeshMaterial3d(material.clone()),
                            Transform::from_xyz(
                                x as f32 * WATER_TOTAL_SIZE,
                                WATER_HEIGHT,
                                z as f32 * WATER_TOTAL_SIZE,
                            ),
                            DebugMaterial("Water"),
                            Water,
                            NotShadowCaster,
                            NotShadowReceiver,
                        ))
                        .id();
                    data.map.insert((x, z), entity);
                }
            }
        }
    }
}

/// use generic bool
/// - true: despawn always
/// - false: only despawn if water is disabled
fn despawn_water<const B: bool>(
    mut commands: Commands,
    water: Query<Entity, With<Water>>,
    settings: Res<ActiveSettingsBank>,
    mut data: ResMut<WaterData>,
) {
    if !B && settings.graphics.water {
        return;
    }
    *data = WaterData::default();
    for e in water.iter() {
        commands.entity(e).despawn_recursive();
    }
}
