use std::f32::consts::PI;

use bevy::{
    asset::Asset,
    core_pipeline::{
        core_3d,
        tonemapping::{DebandDither, Tonemapping},
    },
    math::Vec3A,
    pbr::{Material, MaterialPipeline, MaterialPipelineKey},
    prelude::*,
    reflect::TypePath,
    render::{
        camera::{
            CameraMainTextureUsages, CameraProjection, CameraProjectionPlugin, CameraRenderGraph,
        },
        mesh::{Mesh, MeshVertexBufferLayoutRef, PrimitiveTopology},
        primitives::Frustum,
        render_asset::RenderAssetUsages,
        render_resource::{
            AsBindGroup, Extent3d, RenderPipelineDescriptor, ShaderRef,
            SpecializedMeshPipelineError, TextureDescriptor, TextureDimension, TextureFormat,
            TextureUsages,
        },
        view::{RenderLayers, VisibleEntities},
    },
};

use crate::{
    define_asset_collection, game::asset_loading::LACManager, settings::ActiveSettingsBank,
    ui::in_game::settings::setting_menu_system, AppState,
};

pub const TEXTURE_SIZE: u32 = 512;
const VORONOI_RENDER_LAYER: usize = 4;

const CENTROID_COUNT: usize = 128;
const REPULSION_DISTANCE: f32 = 0.1;
const REPULSION_FORCE: f32 = 0.003;
const CIRCLE_FORCE_FACTOR: f32 = 0.15;
const MOVEMENT_DAMPING: f32 = 0.002;
const RANDOM_WALK_SPEED_FACTOR: f32 = 0.9;

pub struct VoronoiPlugin;

impl Plugin for VoronoiPlugin {
    fn build(&self, app: &mut App) {
        app.register_asset_collection::<VoronoiAsset>();
        app.add_plugins((
            MaterialPlugin::<VoronoiMaterial>::default(),
            CameraProjectionPlugin::<SimpleOrthoProjection>::default(),
        ));
        app.insert_resource(Voronoi::new(CENTROID_COUNT));
        app.add_systems(OnEnter(AppState::InGame), start_voronoi);
        app.add_systems(OnExit(AppState::InGame), stop_voronoi::<true>);
        app.add_systems(
            Update,
            (
                stop_voronoi::<false>,
                start_voronoi.after(setting_menu_system),
            )
                .run_if(resource_changed::<ActiveSettingsBank>)
                .run_if(in_state(AppState::InGame)),
        );
        app.add_systems(Update, update.run_if(in_state(AppState::InGame)));
    }
}

define_asset_collection!(
    VoronoiAsset,
    !shader : Shader = "shaders/voronoi_material.wgsl",
);

fn update(
    mut voronoi_mesh: Query<&Mesh3d, With<VoronoiMesh>>,
    mut meshes: ResMut<Assets<Mesh>>,
    mut voronoi: ResMut<Voronoi>,
    time: Res<Time<Real>>,
) {
    let Ok(mesh) = voronoi_mesh.get_single_mut() else {
        return;
    };
    let Some(mesh) = meshes.get_mut(mesh) else {
        return;
    };
    voronoi.move_r(time.delta().as_secs_f32());
    let (positions, uvs) = voronoi.gen();
    mesh.insert_attribute(Mesh::ATTRIBUTE_POSITION, positions);
    mesh.insert_attribute(Mesh::ATTRIBUTE_UV_0, uvs);
}

#[derive(Component)]
struct VoronoiMesh;

#[derive(Component)]
struct VoronoiCam;

pub fn start_voronoi(
    mut commands: Commands,
    mut meshes: ResMut<Assets<Mesh>>,
    mut materials: ResMut<Assets<VoronoiMaterial>>,
    mut images: ResMut<Assets<Image>>,
    mut voronoi: ResMut<Voronoi>,
    settings: Res<ActiveSettingsBank>,
) {
    if !settings.graphics.water || voronoi.image_handle.is_some() {
        return;
    }
    // create internal Texture
    let size = Extent3d {
        width: TEXTURE_SIZE,
        height: TEXTURE_SIZE,
        ..default()
    };
    let mut image = Image {
        texture_descriptor: TextureDescriptor {
            label: None,
            size,
            dimension: TextureDimension::D2,
            #[cfg(not(target_os = "android"))]
            format: TextureFormat::R32Float,
            #[cfg(target_os = "android")]
            format: TextureFormat::Bgra8UnormSrgb,
            mip_level_count: 1,
            sample_count: 1,
            usage: TextureUsages::TEXTURE_BINDING
                | TextureUsages::COPY_DST
                | TextureUsages::RENDER_ATTACHMENT,
            view_formats: &[],
        },
        ..default()
    };
    image.resize(size);
    let image_handle = images.add(image);

    // setup RenderPass to internal Texture
    let first_pass_layer = RenderLayers::layer(VORONOI_RENDER_LAYER);

    // camera
    commands
        .spawn((
            CameraRenderGraph::new(core_3d::graph::Core3d),
            SimpleOrthoProjection,
            Frustum::from_clip_from_world(&Mat4::IDENTITY),
            Transform::default(),
            GlobalTransform::default(),
            VisibleEntities::default(),
            Camera {
                clear_color: ClearColorConfig::Custom(Color::WHITE),
                order: -2,
                target: image_handle.clone().into(),
                ..default()
            },
            Camera3d::default(),
            Msaa::Off,
            Tonemapping::None,
            DebandDither::Disabled,
            CameraMainTextureUsages::default(),
            first_pass_layer.clone(),
            VoronoiCam,
        ))
        .remove::<Projection>();

    // setup voronoi mesh
    let (positions, uvs) = voronoi.gen();
    let mesh = Mesh::new(
        PrimitiveTopology::TriangleList,
        RenderAssetUsages::default(),
    )
    .with_inserted_attribute(Mesh::ATTRIBUTE_POSITION, positions)
    .with_inserted_attribute(Mesh::ATTRIBUTE_UV_0, uvs);

    // quad
    commands.spawn((
        Mesh3d(meshes.add(mesh)),
        Transform::default(),
        MeshMaterial3d(materials.add(VoronoiMaterial::new(voronoi.r))),
        VoronoiMesh,
        first_pass_layer,
    ));

    // save image_handel
    voronoi.image_handle = Some(image_handle.clone());

    info!("Started rendering voronoi texture");
}

/// use generic bool
/// - true: remove always
/// - false: only remove if water is disabled
fn stop_voronoi<const B: bool>(
    mut commands: Commands,
    cam: Query<Entity, With<VoronoiCam>>,
    mesh: Query<Entity, With<VoronoiMesh>>,
    mut voronoi: ResMut<Voronoi>,
    settings: Res<ActiveSettingsBank>,
) {
    if !B && settings.graphics.water {
        return;
    }
    for e in cam.iter() {
        commands.entity(e).despawn_recursive();
    }
    for e in mesh.iter() {
        commands.entity(e).despawn_recursive();
    }
    if voronoi.image_handle.is_some() {
        info!("Stoped rendering voronoi texture");
    }
    voronoi.image_handle = None;
}

#[derive(Resource)]
pub struct Voronoi {
    pub points: Vec<Particle>,
    pub k: usize,
    pub r: f32,
    pub image_handle: Option<Handle<Image>>,
}

pub struct Particle {
    pub pos: Vec2,
    pub vel: Vec2,
    pub size: f32,
    pub vel2: Vec2,
    pub dir: bool,
}

impl Voronoi {
    pub fn new(k: usize) -> Self {
        let mut points: Vec<Particle> = Vec::new();
        let r = 2.0 * (4.0 / PI / k as f32).sqrt();

        for _ in 0..k {
            points.push(Particle::default());
        }
        Self {
            points,
            k,
            r,
            image_handle: None,
        }
    }

    /// generates positions and uvs for „voronoi-centroid“ mesh.
    /// - positions are triangulated mesh.
    /// - the uvs point to the centroid center and are needed by VoronoiMaterial.
    pub fn gen(&self) -> (Vec<[f32; 3]>, Vec<[f32; 2]>) {
        /// add centroid to mesh
        /// positions are downsized to better create repeating texture
        fn add(quad: &mut Vec<[f32; 3]>, uvs: &mut Vec<[f32; 2]>, r: f32, p: &Vec2) {
            let p = [p[0] * 0.8 + 0.1, p[1] * 0.8 + 0.1];
            let x = p[0] * 2.0 - 1.0;
            let y = -p[1] * 2.0 + 1.0;
            let r = r * 2.0;
            quad.extend_from_slice(&[
                [x + -r, y + -r, 0.0],
                [x + r, y + r, 0.0],
                [x + -r, y + r, 0.0],
                [x + -r, y + -r, 0.0],
                [x + r, y + -r, 0.0],
                [x + r, y + r, 0.0],
            ]);
            uvs.extend(vec![p; 6]);
        }
        let mut quad: Vec<[f32; 3]> = Vec::with_capacity(self.k * 6);
        let mut uvs: Vec<[f32; 2]> = Vec::with_capacity(self.k * 6);
        for p in self.points.iter() {
            // add centroid to mesh
            add(&mut quad, &mut uvs, self.r, &p.pos);

            // add close-to-side centroids to the other side (needed for repeating mesh)
            // sides
            if p.pos.x < self.r {
                add(&mut quad, &mut uvs, self.r, &(p.pos + Vec2::X));
            }
            if p.pos.x > 1. - self.r {
                add(&mut quad, &mut uvs, self.r, &(p.pos - Vec2::X));
            }
            if p.pos.y < self.r {
                add(&mut quad, &mut uvs, self.r, &(p.pos + Vec2::Y));
            }
            if p.pos.y > 1. - self.r {
                add(&mut quad, &mut uvs, self.r, &(p.pos - Vec2::Y));
            }

            //corners
            if p.pos.x < self.r && p.pos.y < self.r {
                add(&mut quad, &mut uvs, self.r, &(p.pos + Vec2::X + Vec2::Y));
            }
            if p.pos.x > 1. - self.r && p.pos.y < self.r {
                add(&mut quad, &mut uvs, self.r, &(p.pos - Vec2::X + Vec2::Y));
            }
            if p.pos.x < self.r && p.pos.y > 1. - self.r {
                add(&mut quad, &mut uvs, self.r, &(p.pos + Vec2::X - Vec2::Y));
            }
            if p.pos.x > 1. - self.r && p.pos.y > 1. - self.r {
                add(&mut quad, &mut uvs, self.r, &(p.pos - Vec2::X - Vec2::Y));
            }
        }
        (quad, uvs)
    }
    /// move centroids
    pub fn move_r(&mut self, delta: f32) {
        for i in 0..self.k {
            for j in 0..self.k {
                if i == j {
                    continue;
                }
                let other = self.points[j].pos;
                let size = self.points[j].size;
                self.points[i].apply_repulsion(other, size, delta);
            }
            let x = fastrand::f32() * 2.0 - 1.0;
            let y = fastrand::f32() * 2.0 - 1.0;
            self.points[i].vel += Vec2::new(x, y) * delta * RANDOM_WALK_SPEED_FACTOR;
            self.points[i].apply_arc(delta);
            self.points[i].apply_vel(delta);
        }
    }
}

impl Default for Particle {
    fn default() -> Self {
        Self {
            pos: Vec2::new(fastrand::f32(), fastrand::f32()),
            vel: Vec2::ZERO,
            size: fastrand::f32() * 2.0 + 0.5,
            vel2: Vec2::new(fastrand::f32() - 0.5, fastrand::f32() - 0.5),
            dir: fastrand::bool(),
        }
    }
}

impl Particle {
    // dampens velocity.
    pub fn apply_vel(&mut self, delta: f32) {
        self.vel *= MOVEMENT_DAMPING.powf(delta);
        self.pos += self.vel * delta;
        self.pos = (self.pos + 1.0) % 1.0;
    }

    /// Applies force to move particle in a circle.
    pub fn apply_arc(&mut self, delta: f32) {
        self.vel2 +=
            Vec2::new(self.vel2.y, -self.vel2.x) * delta * if self.dir { -1.0 } else { 1.0 };
        self.vel2 = self.vel2.normalize_or_zero();
        self.vel += self.vel2 * delta * CIRCLE_FORCE_FACTOR;
    }

    fn apply_repulsion(&mut self, other: Vec2, size: f32, delta: f32) {
        const SIZE: f32 = 1.0;

        // NST of [e^(-x²)]'' is: 2 x² =  A²
        // setting x = REPULSION_DISTANCE
        // use this to scale force

        // delta vector (with repeating world)
        let dpos = (self.pos - other + SIZE / 2.0) % SIZE - SIZE / 2.0;
        let distance = dpos.length_squared();

        if distance < REPULSION_DISTANCE * REPULSION_DISTANCE {
            let force = size
                * REPULSION_FORCE
                * (-distance / (REPULSION_DISTANCE * REPULSION_DISTANCE * 2.0)).exp();
            self.vel += dpos / distance * force * delta;
        }
    }
}

#[derive(Component, Debug, Default, Clone, Reflect)]
#[reflect(Component, Default)]
/// IDENTITY cam projection
pub struct SimpleOrthoProjection;

impl CameraProjection for SimpleOrthoProjection {
    fn get_clip_from_view(&self) -> Mat4 {
        Mat4::IDENTITY
    }
    fn get_clip_from_view_for_sub(&self, _sub_view: &bevy::render::camera::SubCameraView) -> Mat4 {
        Mat4::IDENTITY
    }

    // what to do on window resize
    fn update(&mut self, _width: f32, _height: f32) {}

    fn far(&self) -> f32 {
        1000.
    }

    fn get_frustum_corners(&self, z_near: f32, z_far: f32) -> [bevy::math::Vec3A; 8] {
        let area = Rect::new(-1.0, -1.0, 1.0, 1.0);
        // NOTE: These vertices are in the specific order required by [`calculate_cascade`].
        [
            Vec3A::new(area.max.x, area.min.y, z_near), // bottom right
            Vec3A::new(area.max.x, area.max.y, z_near), // top right
            Vec3A::new(area.min.x, area.max.y, z_near), // top left
            Vec3A::new(area.min.x, area.min.y, z_near), // bottom left
            Vec3A::new(area.max.x, area.min.y, z_far),  // bottom right
            Vec3A::new(area.max.x, area.max.y, z_far),  // top right
            Vec3A::new(area.min.x, area.max.y, z_far),  // top left
            Vec3A::new(area.min.x, area.min.y, z_far),  // bottom left
        ]
    }
}

#[derive(Asset, TypePath, AsBindGroup, Debug, Clone)]
/// Voronoi shader
pub struct VoronoiMaterial {
    #[uniform(0)]
    pub r: Vec4,
}

impl VoronoiMaterial {
    pub fn new(r: f32) -> Self {
        Self {
            r: Vec4::new(r, 0.0, 0.0, 0.0),
        }
    }
}

impl Material for VoronoiMaterial {
    fn vertex_shader() -> ShaderRef {
        "shaders/voronoi_material.wgsl".into()
    }
    fn fragment_shader() -> ShaderRef {
        "shaders/voronoi_material.wgsl".into()
    }

    fn specialize(
        _pipeline: &MaterialPipeline<Self>,
        descriptor: &mut RenderPipelineDescriptor,
        layout: &MeshVertexBufferLayoutRef,
        _key: MaterialPipelineKey<Self>,
    ) -> Result<(), SpecializedMeshPipelineError> {
        let vertex_layout = layout.0.get_layout(&[
            Mesh::ATTRIBUTE_POSITION.at_shader_location(0),
            Mesh::ATTRIBUTE_UV_0.at_shader_location(1),
        ])?;
        descriptor.vertex.buffers = vec![vertex_layout];
        Ok(())
    }
}
