use std::time::Duration;

use bevy::prelude::*;

use crate::{
    game::{
        controls::placer::TileClickEvent,
        map::Map,
        resources::{RelKind, RelevantTileMarker, ResourceCalcSet, TilePlacedEvent},
        tiles::TileType,
    },
    AppState,
};

const HOP_HEIGTH: f32 = 0.2;
const HOP_TIME: f32 = 0.3;
const WAVE_SPEED: f32 = 10.0;

pub struct VisualizeRelevantTilesPlugin;

impl Plugin for VisualizeRelevantTilesPlugin {
    fn build(&self, app: &mut App) {
        app.add_systems(Update, animate.run_if(in_state(AppState::InGame)));
        app.add_systems(
            Update,
            (visualize_placed, visualize_clicked)
                .run_if(in_state(AppState::InGame))
                .after(ResourceCalcSet::CollectRates),
        );
    }
}

impl RelKind {
    pub fn delay(&self) -> f32 {
        match self {
            RelKind::Primary => 0.0,
            RelKind::Secondary => HOP_TIME,
        }
    }
}

#[derive(Debug, Component)]
struct RelevantAnimation {
    pub kind: RelKind,
    pub distance: f32,
    pub timer: Timer,
}

impl RelevantAnimation {
    fn new(kind: RelKind, distance: f32) -> Self {
        Self {
            kind,
            distance,
            timer: Timer::new(
                Duration::from_secs_f32(HOP_TIME + distance / WAVE_SPEED + kind.delay()),
                TimerMode::Once,
            ),
        }
    }
}

fn visualize_placed(
    mut commands: Commands,
    mut event_reader: EventReader<TilePlacedEvent>,
    map: Res<Map>,
    base_tile: Query<&RelevantTileMarker, With<TileType>>,
) {
    for event in event_reader.read() {
        let coord = event.0.coord;
        if let Some(Ok(marker)) = map.get(event.0.coord).map(|entity| base_tile.get(entity)) {
            for (other_coord, kind) in marker.coords.iter() {
                if let Some(entity) = map.get(*other_coord) {
                    commands.entity(entity).insert(RelevantAnimation::new(
                        *kind,
                        Vec3::from(coord - *other_coord).length(),
                    ));
                }
            }
        }
    }
}

fn visualize_clicked(
    mut commands: Commands,
    mut event_reader: EventReader<TileClickEvent>,
    map: Res<Map>,
    base_tile: Query<&RelevantTileMarker, With<TileType>>,
) {
    for event in event_reader.read() {
        if let Some(Ok(marker)) = map.get(event.tile_pos).map(|entity| base_tile.get(entity)) {
            for (other_coord, kind) in marker.coords.iter() {
                if let Some(entity) = map.get(*other_coord) {
                    commands.entity(entity).insert(RelevantAnimation::new(
                        *kind,
                        Vec3::from(event.tile_pos - *other_coord).length(),
                    ));
                }
            }
        }
    }
}

fn animate(
    mut commands: Commands,
    mut tiles: Query<(Entity, &mut Transform, &mut RelevantAnimation), With<RelevantAnimation>>,
    time: Res<Time<Virtual>>,
) {
    for (entity, mut transform, mut marker) in tiles.iter_mut() {
        marker.timer.tick(time.delta());
        if marker.timer.finished() {
            commands.entity(entity).remove::<RelevantAnimation>();
            transform.translation.y = 0.0;
            continue;
        }
        let frac =
            ((marker.timer.elapsed_secs() - (marker.distance / WAVE_SPEED) - marker.kind.delay())
                / HOP_TIME)
                .clamp(0.0, 1.0);
        let height = if frac < 0.5 {
            HOP_HEIGTH * frac * 2.0
        } else {
            HOP_HEIGTH * (1.0 - frac) * 2.0
        };
        transform.translation.y = height;
    }
}
