use std::collections::HashMap;

use bevy::{asset::LoadState, prelude::*};

use crate::{
    coordinates::CubeCoordinate,
    errors::ErrorDisplay,
    game::{
        asset_loading::{AssetLoadStateDump, LoadableAssetCollection},
        rewards::{Reward, RewardType},
    },
    i18n::Translate,
    prelude::LACManager,
    AppState,
};

// optics
const REWARD_RADIAL_ROTATIONAL_SPEED: f32 = 0.5;
const REWARD_RADIAL_TRANSLATIONAL_SPEED: f32 = 1.0;
const REWARD_SIZE: f32 = 0.5;
/// the reward will be spawned with this gap above the tile.
const REWARD_HEIGHT_OFFSET: f32 = 0.3;

pub struct RewardsVisualPlugin;
impl Plugin for RewardsVisualPlugin {
    fn build(&self, app: &mut App) {
        app.register_asset_collection::<RewardsAssetCollection>();
        app.add_systems(
            Update,
            spawn_reward_model
                .run_if(in_state(AppState::InGame))
                .run_if(any_with_component::<Reward>),
        );
        app.add_systems(
            Update,
            animate_reward_model
                .run_if(in_state(AppState::InGame))
                .run_if(any_with_component::<Reward>),
        );
    }
}

impl RewardType {
    /// returns asset paths, scene index and scale factor.
    /// Vector is sorted by asset value (in game).
    fn get_asset_path(&self) -> Vec<(&'static str, usize, f32)> {
        use chrono::Datelike;

        let now = chrono::Utc::now();
        let is_christmas = now.month() > 11 || now.month() < 2;

        match self {
            Self::Food => {
                if is_christmas {
                    vec![
                        ("models/holiday/candy-cane-red.glb", 0, 1.0),
                        ("models/holiday/sock-red-cane.glb", 0, 1.0),
                    ]
                } else {
                    vec![
                        ("models/food/pumpkin.glb", 0, 1.1),
                        ("models/food/wholeHam.glb", 0, 0.9),
                    ]
                }
            }
            Self::Materials => {
                if is_christmas {
                    vec![
                        ("models/lumber-snow.glb", 0, 1.0),
                        ("models/graveyard/debris-snow.glb", 0, 1.0),
                        ("models/holiday/rocks-small.glb", 0, 0.5),
                    ]
                } else {
                    vec![
                        ("models/lumber.glb", 0, 1.0),
                        ("models/graveyard/debris.glb", 0, 1.0),
                        ("models/graveyard/rocks-tall.glb", 0, 0.5),
                    ]
                }
            }
            Self::Money => {
                if is_christmas {
                    vec![
                        ("models/holiday/present-b-cube.glb", 0, 0.9),
                        ("models/holiday/present-b-round.glb", 0, 1.0),
                        ("models/holiday/present-b-rectangle.glb", 0, 1.1),
                    ]
                } else {
                    vec![
                        ("models/platformer/coin-bronze.glb", 0, 1.0),
                        ("models/platformer/coin-silver.glb", 0, 1.0),
                        ("models/platformer/coin-gold.glb", 0, 1.0),
                    ]
                }
            }
        }
    }

    /// retuns random entry of the vector from `self::get_asset_path()`
    #[allow(dead_code)]
    fn get_random_asset(&self) -> (&'static str, usize, f32) {
        let paths = self.get_asset_path();
        let index = fastrand::usize(0..paths.len());
        paths[index]
    }

    /// retuns valued entry of the vector from `self::get_asset_path()`
    /// val should be a value betwenn in range 0.0..1.0
    fn get_valued_asset(&self, val: f32) -> (&'static str, usize, f32) {
        let paths = self.get_asset_path();
        paths[((paths.len() as f32 * val) as usize).clamp(0, paths.len() - 1)]
    }
}

/// add reward model to reward entity
fn spawn_reward_model(
    mut commands: Commands,
    rewards: Query<(Entity, &Reward, &CubeCoordinate), Added<Reward>>,
    reward_models: Res<RewardsAssetCollection>,
    assets: Res<Assets<Gltf>>,
) {
    for (entity, reward, pos) in rewards.iter() {
        let (asset_path, scene, scale) = reward.kind.get_valued_asset(reward.val);

        let model = reward_models
            .0
            .get(asset_path)
            .expect("The LoadingScreen should have ensured, that the asset was requested");
        let gltf = assets
            .get(model)
            .expect("The LoadingScreen should have ensured that the asset was loaded.");
        let scene = gltf
            .scenes
            .get(scene)
            .expect("The model specified that the scene exists.");

        commands.entity(entity).insert((
            SceneRoot(scene.clone_weak()),
            Transform::from_translation(
                Vec3::from(*pos).with_y(reward.base_height + REWARD_HEIGHT_OFFSET),
            )
            .with_scale(Vec3::splat(REWARD_SIZE * scale)),
        ));
    }
}

fn animate_reward_model(time: Res<Time>, mut rewards: Query<(&mut Transform, &Reward)>) {
    let elapsed = time.elapsed_secs();
    let delta = time.delta_secs();
    for (mut pos, reward) in rewards.iter_mut() {
        pos.translation.y = (elapsed * REWARD_RADIAL_TRANSLATIONAL_SPEED).sin() * 0.1
            + reward.base_height
            + REWARD_HEIGHT_OFFSET;
        pos.rotate_y(delta * REWARD_RADIAL_ROTATIONAL_SPEED);
    }
}

#[derive(Default, Resource)]
struct RewardsAssetCollection(HashMap<String, Handle<Gltf>>);

impl LoadableAssetCollection for RewardsAssetCollection {
    fn load_all(asset_server: &AssetServer) -> Self {
        let mut collection = Self::default();

        for reward_type in RewardType::list() {
            for (path, _, _) in reward_type.get_asset_path() {
                // do not reload asset if it was already requested
                if collection.0.contains_key(path) {
                    continue;
                }
                collection
                    .0
                    .insert(path.to_string(), asset_server.load(path));
            }
        }

        collection
    }

    fn check_all(&mut self, asset_server: &AssetServer) -> AssetLoadStateDump {
        let mut dump = AssetLoadStateDump::default();

        for (_, handle) in self.0.iter() {
            dump.requested += 1;
            match asset_server.get_load_state(handle) {
                Some(LoadState::Loaded) => dump.loaded += 1,
                Some(LoadState::Failed(_)) => dump.failed += 1,
                _ => (),
            }
        }

        dump
    }

    fn get_error(
        &self,
        localization: Option<&crate::i18n::Localization>,
    ) -> Option<crate::errors::ErrorDisplay> {
        Some(ErrorDisplay {
            title: localization
                .and_then(|l| l.try_translate("missing-reward-title"))
                .unwrap_or(String::from("Missing reward asset")),
            description: localization
                .and_then(|l| l.try_translate("missing-reward-description"))
                .unwrap_or(String::from(
                    "Without them, finding the rewards will be hard",
                )),
            link: None,
        })
    }
}
