use bevy::{
    prelude::*,
    render::{mesh::PrimitiveTopology, render_asset::RenderAssetUsages},
};
use meshtext::{MeshGenerator, MeshText, TextSection};

use crate::game::asset_loading::LACManager;
use crate::{define_asset_collection, settings::ActiveSettingsBank, AppState};

define_asset_collection!(
    DebugTextAssets,
    !font : Font = "fonts/Fira/FiraMono-Regular.ttf",
);

pub struct DebugTextPlugin;
impl Plugin for DebugTextPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.register_asset_collection::<DebugTextAssets>();
        app.add_systems(
            Update,
            (spawn_debug_text, mark_debug_text, despawn_debug_text)
                .run_if(in_state(AppState::InGame)),
        );
        app.add_systems(
            Update,
            update_debug_text_settings
                .before(spawn_debug_text)
                .run_if(resource_changed::<ActiveSettingsBank>)
                .run_if(in_state(AppState::InGame)),
        );
    }
}

#[derive(Component)]
pub struct DebugText {
    pub text: String,
    pub font_size: f32,
    pub height: f32,
    pub render: bool,
    pub was_rendered: bool,
}

impl DebugText {
    pub fn new(text: String, font_size: f32) -> Self {
        Self {
            text,
            font_size,
            height: 0.0,
            render: true,
            was_rendered: false,
        }
    }

    #[allow(dead_code)]
    pub fn with_height(mut self, height: f32) -> Self {
        self.height = height;
        self
    }

    /// Add new section. Deletes all, if content was shown before.
    /// name: should not have line-break
    #[allow(dead_code)]
    pub fn add_section(&mut self, name: &str, str: String) {
        if self.was_rendered {
            self.text.clear();
            self.was_rendered = false;
        }
        self.text.push_str(name);
        self.text.push('\n');
        for line in str.split('\n') {
            self.text.push_str("    ");
            self.text.push_str(line);
            self.text.push('\n');
        }
    }

    pub fn set_text(&mut self, text: String) {
        self.text = text;
    }

    /// Needs one game tick to delete child entities.
    #[allow(dead_code)]
    pub fn destroy(&mut self) {
        self.render = false;
    }
}

#[derive(Debug, Component)]
pub struct DebugTextAlways;

#[derive(Component)]
pub struct DebugTextMarker;

/// should not be used by the user
#[derive(Component)]
pub struct TextEntityData {
    handle: Handle<Mesh>,
}

impl TextEntityData {
    fn new(handle: Handle<Mesh>) -> Self {
        Self { handle }
    }
}

#[allow(clippy::type_complexity)]
pub fn despawn_debug_text(
    mut commands: Commands,
    mut meshes: ResMut<Assets<Mesh>>,
    tiles: Query<(Entity, &Children), (Without<DebugText>, With<DebugTextMarker>)>,
    text_entities: Query<(Entity, &TextEntityData)>,
) {
    for (e_id, children) in tiles.iter() {
        for child in children.iter() {
            if let Ok((child_id, child_data)) = text_entities.get(*child) {
                meshes.remove(&child_data.handle);
                commands.entity(child_id).despawn_recursive();
            }
        }
        commands
            .entity(e_id)
            .remove::<DebugTextMarker>()
            .remove::<DebugTextAlways>();
    }
}

pub fn mark_debug_text(mut commands: Commands, tiles: Query<Entity, Added<DebugText>>) {
    for e_id in tiles.iter() {
        commands.entity(e_id).insert(DebugTextMarker);
    }
}

#[allow(clippy::type_complexity)]
pub fn update_debug_text_settings(
    settings: Res<ActiveSettingsBank>,
    mut commands: Commands,
    mut tiles: Query<(&Children, &mut DebugText), (With<DebugText>, Without<DebugTextAlways>)>,
    text_entities: Query<&TextEntityData>,
    mut meshes: ResMut<Assets<Mesh>>,
) {
    if !settings.graphics.debug_text {
        for (childs, _) in tiles.iter() {
            for child in childs {
                if let Ok(child_data) = text_entities.get(*child) {
                    meshes.remove(&child_data.handle);
                    commands.entity(*child).despawn_recursive();
                }
            }
        }
    } else {
        for (_, mut debug_text) in tiles.iter_mut() {
            debug_text.text += "";
        }
    }
}

#[allow(clippy::too_many_arguments)]
pub fn spawn_debug_text(
    mut commands: Commands,
    mut meshes: ResMut<Assets<Mesh>>,
    mut materials: ResMut<Assets<StandardMaterial>>,
    mut text_material: Local<Option<Handle<StandardMaterial>>>,
    text_entities: Query<&TextEntityData>,
    tiles: Query<(Entity, &DebugText, Option<&DebugTextAlways>), Changed<DebugText>>,
    tiles_childs: Query<&Children, Changed<DebugText>>,
    settings: Res<ActiveSettingsBank>,
    handles: Res<DebugTextAssets>,
    font_assets: ResMut<Assets<Font>>,
) {
    let tiles = tiles
        .iter()
        .filter(|(_, _, always)| settings.graphics.debug_text || always.is_some());

    let font_data = font_assets
        .get(&handles.font)
        .expect("The asset loader loaded the asset");

    let mut generator = MeshGenerator::new(font_data.data.to_vec());
    for (e_id, text, _) in tiles {
        // despawn former Text
        if let Ok(childs) = tiles_childs.get(e_id) {
            for child in childs {
                if let Ok(child_data) = text_entities.get(*child) {
                    meshes.remove(&child_data.handle);
                    commands.entity(*child).despawn_recursive();
                }
            }
        }

        if !text.render || text.text.is_empty() {
            continue;
        }

        // Add new Text
        commands.entity(e_id).with_children(|e| {
            let mut pos: f32 = 0.0;
            for line in text.text.split('\n') {
                let text_mesh: MeshText = generator
                    .generate_section(line, true, None)
                    .expect("Expected MeshGenerator to generate MeshText");

                let vertices = text_mesh.vertices;
                let positions: Vec<[f32; 3]> =
                    vertices.chunks(3).map(|c| [c[0], c[1], c[2]]).collect();

                let mesh = Mesh::new(
                    PrimitiveTopology::TriangleList,
                    RenderAssetUsages::default(),
                )
                .with_inserted_attribute(Mesh::ATTRIBUTE_POSITION, positions)
                .with_computed_flat_normals();

                let mesh_h = meshes.add(mesh);
                e.spawn((
                    Mesh3d(mesh_h.clone()),
                    MeshMaterial3d(
                        text_material
                            .get_or_insert_with(|| materials.add(Color::srgb(1.0, 0.0, 0.0)))
                            .clone(),
                    ),
                    Transform::from_xyz(0.4 - pos, text.height + 0.20001, -0.4)
                        .with_rotation(Quat::from_xyzw(-0.5, -0.5, -0.5, 0.5))
                        .with_scale(Vec3::new(text.font_size * 2.0 / 3.0, text.font_size, 0.1)),
                    TextEntityData::new(mesh_h),
                ));
                pos += text.font_size * 1.0;
            }
        });
    }
}

pub fn debug_text_tick(mut tiles: Query<&mut DebugText, Changed<DebugText>>) {
    for mut text in tiles.iter_mut() {
        text.was_rendered = true;
    }
}
