use bevy::{
    core_pipeline::{
        core_2d,
        tonemapping::{DebandDither, Tonemapping},
    },
    prelude::*,
    render::{
        camera::{CameraMainTextureUsages, CameraRenderGraph},
        mesh::{Mesh, PrimitiveTopology},
        primitives::Frustum,
        render_asset::RenderAssetUsages,
        render_resource::{
            AsBindGroup, Extent3d, ShaderRef, TextureDescriptor, TextureDimension, TextureFormat,
            TextureUsages,
        },
        view::{RenderLayers, VisibleEntities},
    },
    sprite::{Material2d, Material2dPlugin},
};

use crate::{define_asset_collection, prelude::*, settings::ActiveSettingsBank, AppState};

use super::voronoi::{start_voronoi, SimpleOrthoProjection, Voronoi, TEXTURE_SIZE};

const CELL_WATER_RENDER_LAYER: usize = 3;
const WATER_COLOR: Color = Color::linear_rgba(0.0, 0.0, 1.0, 0.9);
const RIPPLE_COLOR: Color = Color::linear_rgba(1.0, 1.0, 1.0, 1.0);

pub struct CellWaterPlugin;

impl Plugin for CellWaterPlugin {
    fn build(&self, app: &mut App) {
        app.register_asset_collection::<CellWaterAsset>();
        app.add_plugins(Material2dPlugin::<CellWaterMaterial>::default());
        app.insert_resource(CellWater::default());

        app.add_systems(
            OnEnter(AppState::InGame),
            start_cell_water.after(start_voronoi),
        );
        app.add_systems(
            Update,
            (
                stop_cell_water::<false>,
                start_cell_water.after(start_voronoi),
            )
                .run_if(resource_changed::<ActiveSettingsBank>)
                .run_if(in_state(AppState::InGame)),
        );
        app.add_systems(OnExit(AppState::InGame), stop_cell_water::<true>);
    }
}

define_asset_collection!(
    CellWaterAsset,
    !shader : Shader = "shaders/cell_water_material.wgsl",
);

#[derive(Component)]
struct CellWaterMesh;

#[derive(Component)]
struct CellWaterCam;

pub fn start_cell_water(
    mut commands: Commands,
    mut meshes: ResMut<Assets<Mesh>>,
    mut materials: ResMut<Assets<CellWaterMaterial>>,
    mut images: ResMut<Assets<Image>>,
    voronoi: Res<Voronoi>,
    mut cell_water: ResMut<CellWater>,
    settings: Res<ActiveSettingsBank>,
) {
    if !settings.graphics.water || cell_water.image_handle.is_some() {
        return;
    }
    let Some(ref voronoi_handle) = voronoi.image_handle else {
        error!("Started cell water, but voronoi texture was not ready!");
        return;
    };
    // create internal Texture
    let size = Extent3d {
        width: TEXTURE_SIZE,
        height: TEXTURE_SIZE,
        ..default()
    };
    let mut image = Image {
        texture_descriptor: TextureDescriptor {
            label: None,
            size,
            dimension: TextureDimension::D2,
            format: TextureFormat::bevy_default(),
            mip_level_count: 1,
            sample_count: 1,
            usage: TextureUsages::TEXTURE_BINDING
                | TextureUsages::COPY_DST
                | TextureUsages::RENDER_ATTACHMENT,
            view_formats: &[],
        },
        ..default()
    };
    image.resize(size);
    let image_handle = images.add(image);

    // setup RenderPass to internal Texture
    let first_pass_layer = RenderLayers::layer(CELL_WATER_RENDER_LAYER);

    // camera
    commands
        .spawn((
            CameraRenderGraph::new(core_2d::graph::Core2d),
            SimpleOrthoProjection,
            Frustum::from_clip_from_world(&Mat4::IDENTITY),
            Transform::default(),
            GlobalTransform::default(),
            VisibleEntities::default(),
            Camera {
                clear_color: ClearColorConfig::Custom(Color::linear_rgba(1.0, 1.0, 1.0, 0.0)),
                order: -1,
                target: image_handle.clone().into(),
                ..default()
            },
            Camera2d,
            Msaa::Off,
            Tonemapping::None,
            DebandDither::Disabled,
            CameraMainTextureUsages::default(),
            first_pass_layer.clone(),
            CellWaterCam,
        ))
        .remove::<OrthographicProjection>();

    // setup rect with clipped edge to get repeating texture
    let positions: Vec<[f32; 3]> = vec![
        [-1.0, -1.0, 0.0],
        [1.0, 1.0, 0.0],
        [-1.0, 1.0, 0.0],
        [-1.0, -1.0, 0.0],
        [1.0, -1.0, 0.0],
        [1.0, 1.0, 0.0],
    ];
    let uvs: Vec<[f32; 2]> = vec![
        [0.1, 0.1],
        [0.9, 0.9],
        [0.1, 0.9],
        [0.1, 0.1],
        [0.9, 0.1],
        [0.9, 0.9],
    ];

    let mesh = Mesh::new(
        PrimitiveTopology::TriangleList,
        RenderAssetUsages::default(),
    )
    .with_inserted_attribute(Mesh::ATTRIBUTE_POSITION, positions)
    .with_inserted_attribute(Mesh::ATTRIBUTE_UV_0, uvs);

    // quad
    commands.spawn((
        Mesh2d(meshes.add(mesh)),
        Transform::default(),
        MeshMaterial2d(materials.add(CellWaterMaterial {
            water_color: WATER_COLOR.to_linear(),
            ripple_color: RIPPLE_COLOR.to_linear(),
            color_texture: Some(voronoi_handle.clone()),
        })),
        first_pass_layer,
        CellWaterMesh,
    ));

    //save image_handel
    cell_water.image_handle = Some(image_handle);

    info!("Started rendering cell_water texture");
}

/// use generic bool
/// - true: remove always
/// - false: only remove if water is disabled
fn stop_cell_water<const B: bool>(
    mut commands: Commands,
    cam: Query<Entity, With<CellWaterCam>>,
    mesh: Query<Entity, With<CellWaterMesh>>,
    mut cell_water: ResMut<CellWater>,
    settings: Res<ActiveSettingsBank>,
) {
    if !B && settings.graphics.water {
        return;
    }
    for e in cam.iter() {
        commands.entity(e).despawn_recursive();
    }
    for e in mesh.iter() {
        commands.entity(e).despawn_recursive();
    }
    if cell_water.image_handle.is_some() {
        info!("Stoped rendering cell_water texture");
    }
    cell_water.image_handle = None;
}

#[derive(Resource, Default)]
pub struct CellWater {
    pub image_handle: Option<Handle<Image>>,
}

#[derive(Asset, TypePath, AsBindGroup, Debug, Clone)]
pub struct CellWaterMaterial {
    #[uniform(0)]
    water_color: LinearRgba,
    #[uniform(1)]
    ripple_color: LinearRgba,
    #[texture(2)]
    #[sampler(3)]
    color_texture: Option<Handle<Image>>,
}

impl Material2d for CellWaterMaterial {
    fn fragment_shader() -> ShaderRef {
        "shaders/cell_water_material.wgsl".into()
    }
}
