use std::time::Duration;

#[cfg(feature = "graphics")]
use crate::{settings::ActiveSettingsBank, ui::in_game::pause::PauseUiState};
use bevy::prelude::*;
#[cfg(feature = "graphics")]
use bevy::{time::time_system, window::WindowFocused};

use crate::{AppState, DisplayMode};

use super::GamePauseState;

pub struct TimePlugin {
    pub display: DisplayMode,
}

impl Plugin for TimePlugin {
    fn build(&self, app: &mut App) {
        app.add_event::<StepEvent>();
        app.init_resource::<Time<Game>>();
        app.add_systems(OnEnter(AppState::InGame), reset_game_time);
        app.add_systems(OnExit(AppState::InGame), reset_game_time);
        app.add_systems(OnExit(GamePauseState::Running), pause_game_time);
        app.add_systems(OnEnter(GamePauseState::Running), unpause_game_time);
        if let DisplayMode::Headless = self.display {
            app.add_systems(PreUpdate, step_time.run_if(in_state(AppState::InGame)));
        }
        #[cfg(feature = "graphics")]
        if let DisplayMode::Graphic = self.display {
            app.add_systems(First, update_real_game_time.after(time_system));
        }
    }
}

#[derive(Debug, Default)]
pub struct Game {
    paused: bool,
    #[allow(dead_code)]
    time: Duration,
}

#[allow(dead_code)]
pub trait GameTime {
    fn pause(&mut self);
    fn unpause(&mut self);
    fn is_paused(&self) -> bool;
    fn sub_nanos(&self) -> u32;
    fn sub_secs(&self) -> f32;
}

impl GameTime for Time<Game> {
    /// Stops the clock, preventing it from advancing until resumed.
    #[inline]
    fn pause(&mut self) {
        self.context_mut().paused = true;
    }

    /// Resumes the clock if paused.
    #[inline]
    fn unpause(&mut self) {
        self.context_mut().paused = false;
    }

    /// Returns `true` if the clock is currently paused.
    #[inline]
    fn is_paused(&self) -> bool {
        self.context().paused
    }

    #[inline]
    fn sub_nanos(&self) -> u32 {
        self.context().time.subsec_nanos()
    }

    #[inline]
    fn sub_secs(&self) -> f32 {
        (self.sub_nanos() as f64 / 1_000_000_000f64) as f32
    }
}

// the following systems stop vtime too.
// this is necessary, because of the tile loading animation.

pub fn pause_game_time(mut time: ResMut<Time<Game>>, mut vtime: ResMut<Time<Virtual>>) {
    time.pause();
    vtime.pause();
}

pub fn unpause_game_time(mut time: ResMut<Time<Game>>, mut vtime: ResMut<Time<Virtual>>) {
    time.unpause();
    vtime.unpause();
}

pub fn reset_game_time(mut time: ResMut<Time<Game>>, mut vtime: ResMut<Time<Virtual>>) {
    *time = Time::default();
    *vtime = Time::default();
    vtime.set_max_delta(Duration::from_millis(100));
}

/// Should only be used in headless mode, because vtime is not affected by this.
#[cfg(feature = "graphics")]
fn update_real_game_time(mut time: ResMut<Time<Game>>, real: Res<Time<Virtual>>) {
    if !time.context_mut().paused {
        let last_secs = time.context_mut().time.as_secs();
        time.context_mut().time += real.delta();

        let delta_secs = time.context_mut().time.as_secs() - last_secs;
        if delta_secs > 0 {
            if delta_secs > 1 {
                warn!("Could not keep up with the simulation. Graphics will be out of sync.");
            }
            time.advance_by(Duration::from_secs(1));
            return;
        }
    }
    time.advance_by(Duration::ZERO);
}

#[derive(Debug, Event, Deref, DerefMut)]
/// Only available in Headless mode.
/// Steps the time forward about the given amount.
pub struct StepEvent(pub Duration);

pub fn step_time(mut steps: EventReader<StepEvent>, mut time: ResMut<Time<Game>>) {
    let mut total_next_step = Duration::ZERO;
    for step in steps.read() {
        total_next_step += **step;
    }
    if !time.context_mut().paused {
        time.advance_by(total_next_step);
    }
}

#[cfg(feature = "graphics")]
pub fn pause_on_focus_loss(
    mut ev: EventReader<WindowFocused>,
    settings: Res<ActiveSettingsBank>,
    curr: Res<State<GamePauseState>>,
    mut pause: ResMut<NextState<GamePauseState>>,
    mut pause_ui: ResMut<NextState<PauseUiState>>,
) {
    // player disabled auto-pause
    if !settings.game_behaviour.pause_when_unfocused {
        ev.clear();
        return;
    }
    // only pause the game on unfocus
    // everything else doesn't matter
    if curr.get() != &GamePauseState::Running {
        ev.clear();
        return;
    }

    for e in ev.read() {
        if !e.focused {
            pause.set(GamePauseState::Paused);
            pause_ui.set(PauseUiState::Shown);
        }
    }
}
