use crate::{
    game::{tiles::CubeCoordinate, GameSimSet, Map},
    AppState,
};
use bevy::prelude::*;
use serde::{Deserialize, Serialize};

pub struct SoundTilePlugin {
    #[cfg(feature = "graphics")]
    pub display_mode: crate::DisplayMode,
}

impl Plugin for SoundTilePlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.add_event::<ChangeSoundTileData>();
        app.add_event::<SoundTileChangedEvent>();

        app.add_systems(
            Update,
            update_tile_data
                .run_if(in_state(AppState::InGame))
                .in_set(GameSimSet::Prepare),
        );

        #[cfg(feature = "graphics")]
        if let crate::DisplayMode::Graphic = self.display_mode {
            sound::init_app(app);
        }
    }
}

#[derive(Asset, TypePath, Component, Clone, Debug, Serialize, Deserialize)]
pub struct SoundTileData {
    pub enabled: bool,
    /// multiply the wave accumulative output by a factor
    pub amplify: f32,
    /// waves which are played
    pub waves: Vec<Wave>,
}

impl Default for SoundTileData {
    fn default() -> Self {
        Self {
            amplify: 1.0,
            enabled: false,
            waves: vec![Wave::default()],
        }
    }
}

impl SoundTileData {
    pub fn off() -> SoundTileData {
        SoundTileData {
            enabled: false,
            amplify: 0.0,
            waves: Vec::new(),
        }
    }
}

#[derive(Serialize, Deserialize, Clone, Debug)]
pub struct Wave {
    /// wave frequency in Hz
    pub frequency: f32,
    /// waveform used to generate the sound
    pub waveform: Waveform,
    /// enable sound output
    /// the sound generator will return 0 if false
    pub enabled: bool,
    /// phase offset [0,1]
    pub offset: Option<f32>,
    /// factor used to multiply wave
    pub amplify: f32,
}

impl Default for Wave {
    fn default() -> Self {
        Self {
            frequency: 440.0,
            waveform: Default::default(),
            enabled: true,
            offset: None,
            amplify: 1.0,
        }
    }
}

#[derive(Clone, Copy, Debug, Serialize, Deserialize, Default, PartialEq)]
pub enum Waveform {
    /// https://en.wikipedia.org/wiki/Sine_wave
    #[default]
    Sine,
    /// https://en.wikipedia.org/wiki/Sawtooth_wave
    Sawtooth,
    /// https://en.wikipedia.org/wiki/Square_wave_(waveform)
    Square,
    /// https://en.wikipedia.org/wiki/Triangle_wave
    Triangle,
}
impl Waveform {
    pub fn list() -> [Self; 4] {
        [Self::Sine, Self::Sawtooth, Self::Square, Self::Triangle]
    }
}

#[cfg(feature = "graphics")]
impl crate::prelude::WithTranslationID for Waveform {
    fn get_translation_id(&self) -> &str {
        match self {
            Self::Sine => "waveform-sine",
            Self::Sawtooth => "waveform-sawtooth",
            Self::Square => "waveform-square",
            Self::Triangle => "waveform-triangle",
        }
    }
}

#[derive(Debug, Serialize, Deserialize, Clone, Event)]
pub struct ChangeSoundTileData {
    pub coord: CubeCoordinate,
    pub data: SoundTileData,
}

#[derive(Debug, Serialize, Deserialize, Clone, Event)]
pub struct SoundTileChangedEvent(pub CubeCoordinate);

/// update the tile components from the ChangeSoundTileColor event
pub fn update_tile_data(
    mut tiles: Query<(Entity, &mut SoundTileData, &CubeCoordinate)>,
    map: Res<Map>,
    mut events: EventReader<ChangeSoundTileData>,
    mut succ: EventWriter<SoundTileChangedEvent>,
) {
    for ChangeSoundTileData { coord, data } in events.read() {
        if let Some(entity) = map.get(*coord) {
            if let Ok((_, mut tile_data, _)) = tiles.get_mut(entity) {
                *tile_data = data.clone();
                succ.send(SoundTileChangedEvent(*coord));
            }
        }
    }
}

#[cfg(feature = "graphics")]
pub mod sound {

    use std::{
        collections::HashMap,
        f32::consts::PI,
        sync::{
            mpsc::{channel, Receiver, Sender, TryRecvError},
            RwLock,
        },
        time::Duration,
    };

    use bevy::{
        audio::{AddAudioSource, Source, Volume},
        prelude::*,
    };

    use crate::{game::GamePauseState, settings::ActiveSettingsBank, AppState};

    use super::{SoundTileData, Wave, Waveform};

    pub fn init_app(app: &mut App) {
        app.add_audio_source::<TileAudioController>();
        app.add_systems(Startup, setup_audio);
        app.add_systems(
            Update,
            update_audio
                .run_if(in_state(AppState::InGame))
                .run_if(in_state(GamePauseState::Running)),
        );

        app.add_systems(
            Update,
            sync_volume
                .run_if(resource_changed::<ActiveSettingsBank>)
                .run_if(in_state(AppState::InGame)),
        );
        app.add_systems(OnEnter(GamePauseState::Paused), pause_sound);
        app.add_systems(OnEnter(GamePauseState::Running), unpause_sound);
    }

    /// spawns AudioPlayer component / entity, that handles all sound tile output.
    fn setup_audio(
        mut assets: ResMut<Assets<TileAudioController>>,
        mut commands: Commands,
        settings: Res<ActiveSettingsBank>,
    ) {
        let audio_handle = assets.add(TileAudioController::default());
        commands.spawn((
            AudioPlayer(audio_handle),
            PlaybackSettings::default().with_volume(Volume::new(settings.audio.get_sfx_percent())),
        ));
    }

    #[allow(clippy::type_complexity)]
    fn update_audio(
        ctrl: Query<&AudioPlayer<TileAudioController>>,
        assets: Res<Assets<TileAudioController>>,
        tiles: Query<(Entity, &SoundTileData), Or<(Changed<SoundTileData>, Added<SoundTileData>)>>,
        mut removed: RemovedComponents<SoundTileData>,
    ) {
        let Some(ctrl) = ctrl
            .get_single()
            .ok()
            .and_then(|id| assets.get(&id.0))
            .and_then(|obj| obj.sender.read().ok())
        else {
            return;
        };
        let Some(sender) = ctrl.as_ref() else {
            return;
        };

        for (entity, tile) in tiles.iter() {
            let res = sender.send(AudioControllerMessage::UpdateEntity(entity, tile.clone()));
            if res.is_err() {
                warn!("Sound Tile audio has failed to connect to the wave generator.");
            }
        }
        for entity in removed.read() {
            let res = sender.send(AudioControllerMessage::UpdateEntity(
                entity,
                SoundTileData::off(),
            ));
            if res.is_err() {
                warn!("Sound Tile audio has failed to connect to the wave generator.");
            }
        }
    }

    fn pause_sound(
        ctrl: Query<&AudioPlayer<TileAudioController>>,
        assets: Res<Assets<TileAudioController>>,
    ) {
        let Some(ctrl) = ctrl
            .get_single()
            .ok()
            .and_then(|id| assets.get(&id.0))
            .and_then(|obj| obj.sender.read().ok())
        else {
            return;
        };
        let Some(ctrl) = ctrl.as_ref() else {
            return;
        };
        let res = ctrl.send(AudioControllerMessage::AllOff);
        if res.is_err() {
            warn!("Sound Tile audio has failed to connect to the wave generator.");
        }
    }
    fn unpause_sound(
        ctrl: Query<(&AudioPlayer<TileAudioController>, &AudioSink)>,
        assets: Res<Assets<TileAudioController>>,
        tiles: Query<(Entity, &SoundTileData)>,
        settings: Res<ActiveSettingsBank>,
    ) {
        let Ok((player, sink)) = ctrl.get_single() else {
            return;
        };
        sink.set_volume(settings.audio.get_sfx_percent());
        let Some(sender_guard) = assets
            .get(&player.0)
            .and_then(|controller| controller.sender.read().ok())
        else {
            return;
        };
        let Some(sender) = sender_guard.as_ref() else {
            return;
        };

        for (entity, tile) in tiles.iter() {
            let res = sender.send(AudioControllerMessage::UpdateEntity(entity, tile.clone()));
            if res.is_err() {
                warn!("Sound Tile audio has failed to connect to the wave generator. Failed to start the audio.");
            }
        }
    }

    fn sync_volume(
        ctrl: Query<&AudioSink, With<AudioPlayer<TileAudioController>>>,
        settings: Res<ActiveSettingsBank>,
    ) {
        let Ok(sink) = ctrl.get_single() else {
            return;
        };
        sink.set_volume(settings.audio.get_sfx_percent());
    }

    #[derive(TypePath, Asset, Default)]
    /// Controller for all sound tile related audio.
    /// This component will be used like a resource.
    /// The controller holds a channel sender, to control the audio generator.
    struct TileAudioController {
        sender: RwLock<Option<Sender<AudioControllerMessage>>>,
    }

    // Generates audio generator.
    impl Decodable for TileAudioController {
        type DecoderItem = <TileDecoder as Iterator>::Item;

        type Decoder = TileDecoder;

        fn decoder(&self) -> Self::Decoder {
            TileDecoder::new(self)
        }
    }

    /// Messages send to the TileDecoder from the game world.
    enum AudioControllerMessage {
        UpdateEntity(Entity, SoundTileData),
        AllOff,
    }

    /// This decoder is responsible for playing the audio,
    /// and so stores data about the audio being played.
    pub struct TileDecoder {
        data: HashMap<Entity, (SoundTileData, Vec<WaveMeta>)>,
        recv: Receiver<AudioControllerMessage>,
        counter: u32,
        progress: f64,
    }

    impl TileDecoder {
        pub const SAMPLE_RATE: u32 = 44800;
        const PROGRESS_PER_FRAME: f32 = 1.0 / Self::SAMPLE_RATE as f32;
        const PROGRESS_PER_FRAME_F64: f64 = 1.0 / Self::SAMPLE_RATE as f64;
        const RECV_COUNTDOWN: u32 = Self::SAMPLE_RATE / 60;

        fn new(ctrl: &TileAudioController) -> Self {
            let (send, recv) = channel();
            let mut sender = ctrl
                .sender
                .write()
                .expect("TileAudioController lock is poisoned. Unable to save channel sender.");
            sender.replace(send);
            TileDecoder {
                data: HashMap::new(),
                recv,
                counter: 0,
                progress: 0.0,
            }
        }

        fn housekeeping(&mut self) -> Option<()> {
            self.counter += 1;
            if self.counter < Self::RECV_COUNTDOWN {
                return Some(());
            }
            self.counter = 0;
            // try receive new sound updates
            loop {
                match self.recv.try_recv() {
                    Ok(AudioControllerMessage::UpdateEntity(entity, new_data)) => {
                        self.data
                            .entry(entity)
                            .and_modify(|(data, meta)| {
                                *data = new_data.clone();
                                if meta.len() < data.waves.len() {
                                    meta.extend(vec![
                                        WaveMeta::default();
                                        data.waves.len() - meta.len()
                                    ]);
                                }
                            })
                            .or_insert_with(|| {
                                (
                                    new_data.clone(),
                                    (0..new_data.waves.len())
                                        .map(|_| WaveMeta::default())
                                        .collect::<Vec<_>>(),
                                )
                            });
                    }
                    Ok(AudioControllerMessage::AllOff) => {
                        for (data, _) in self.data.values_mut() {
                            *data = SoundTileData::off();
                        }
                    }
                    Err(TryRecvError::Disconnected) => return None,
                    Err(TryRecvError::Empty) => break,
                }
            }

            // cleanup stopped wave channels
            self.data.retain(|_, (waves, meta)| {
                while waves.waves.len() < meta.len()
                    && meta
                        .last()
                        .map(|e| e.y.abs() < f32::EPSILON && e.dy.abs() < f32::EPSILON)
                        .unwrap_or(false)
                {
                    meta.pop();
                }
                !meta.is_empty()
            });
            Some(())
        }
    }

    // The decoder must implement iterator so that it can implement `Decodable`.
    impl Iterator for TileDecoder {
        type Item = f32;

        fn next(&mut self) -> Option<Self::Item> {
            self.housekeeping()?;

            self.progress += Self::PROGRESS_PER_FRAME_F64;

            Some(self.data.values_mut().fold(0.0, |acc, (waves, meta)| {
                acc + meta
                    .iter_mut()
                    .enumerate()
                    .fold(0.0, |acc, (wave_id, meta)| {
                        // use wave settings if enabled, otherwise use None aka. off
                        let wave = waves.waves.get(wave_id).and_then(|w| {
                            if w.enabled && waves.enabled {
                                Some(w)
                            } else {
                                None
                            }
                        });
                        acc + meta.get(wave, waves.amplify, self.progress)
                    })
            }))
        }
    }
    // `Source` is what allows the audio source to be played by bevy.
    // This trait provides information on the audio.
    impl Source for TileDecoder {
        fn current_frame_len(&self) -> Option<usize> {
            None
        }

        fn channels(&self) -> u16 {
            1
        }

        fn sample_rate(&self) -> u32 {
            Self::SAMPLE_RATE
        }

        fn total_duration(&self) -> Option<Duration> {
            None
        }
    }

    /// This wave data describes the current state of the wave. It will be used to modulate the next wave sample.
    #[derive(Default, Clone)]
    struct WaveMeta {
        effective_amplify: f32,
        last_phase_err: f32,
        y: f32,
        dy: f32,
    }

    impl WaveMeta {
        fn get(&mut self, wave: Option<&Wave>, group_amplify: f32, progress: f64) -> f32 {
            let Some(wave) = wave else {
                // smooth transition to off state: 0.0
                self.dy = self.dy * 0.9 + -self.y.signum() * 100.0 * 0.1;
                let old = self.y;

                self.y += self.dy * TileDecoder::PROGRESS_PER_FRAME;

                if self.y * old < 0.0 {
                    self.y = 0.0;
                    self.dy = 0.0;
                }

                return self.y * self.effective_amplify;
            };

            // progress in the current phase. in range [0,1]
            let mut phase_prog = 0.0;

            // do phase adjustment
            let mut phase_adjustment = wave.offset.is_some();

            // target frequency with adjustment for phase error
            let mut frequency = wave.frequency;
            if phase_adjustment {
                frequency *= (self.last_phase_err * 0.5).clamp(-0.25, 0.25) + 1.0;
            }

            match wave.waveform {
                Waveform::Sine => {
                    // omega, common scaling factor
                    let ome = 2.0 * PI * frequency;

                    // ode simulating a harmonic resonance: sin(2*pi*freq * time)
                    // derivative changes with the inverse deflection.
                    self.y += self.dy * TileDecoder::PROGRESS_PER_FRAME;
                    self.dy += -self.y * ome.powi(2) * TileDecoder::PROGRESS_PER_FRAME;

                    self.y = self.y.clamp(-1.0, 1.0);

                    // calculate current phase for phase and amplitude correction
                    // phase in radiant based on the current elongation
                    let mut phase = self.y.asin();
                    // mirror asin to include the second half of the curve
                    if self.dy < 0.0 {
                        phase = PI - phase;
                    }
                    // simple version of phase mod 2PI
                    if phase < 0.0 {
                        phase += 2.0 * PI;
                    }

                    // convert to [0,1] range for phase adjustment
                    if phase_adjustment {
                        phase_prog = phase / (2.0 * PI);
                    }

                    // amplitude correction
                    // calculates theoretic dy and applies it with a smoothing filter to the real dy.
                    let correction = phase.cos() * ome;
                    self.dy = self.dy * 0.99 + correction * 0.01;
                }
                Waveform::Sawtooth => {
                    // ode that creates a sawtooth curve
                    // value range is [-1,1], so the derivative needs to be twice the frequency.
                    let goal = frequency * 2.0;
                    self.dy = self.dy * 0.9 + goal * 0.1;
                    self.y += self.dy * TileDecoder::PROGRESS_PER_FRAME;
                    if self.y > 1.0 {
                        self.y -= 2.0;
                    }
                    if phase_adjustment {
                        phase_prog = (self.y + 1.0) / 2.0;
                    }
                }
                Waveform::Square => {
                    // no phase adjustment needed. Global timer will be used anyway.
                    phase_adjustment = false;
                    self.dy = 0.0;
                    let phase_len = 1.0 / wave.frequency;
                    let square_time = progress as f32 % phase_len;
                    self.y = if square_time * 2.0 < phase_len {
                        1.0
                    } else {
                        -1.0
                    };
                }
                Waveform::Triangle => {
                    // ode that creates a triangle curve
                    // value range is [-1,1] and the slope has to be done twice per cycle
                    // the the absolut of the derivative needs to be 4 times the frequency.
                    let mut goal = frequency * 4.0;
                    if self.dy < 0.0 {
                        goal = -goal;
                    }
                    self.dy = self.dy * 0.9 + goal * 0.1;

                    self.y += self.dy * TileDecoder::PROGRESS_PER_FRAME;
                    // when limit is reached toggle the direction
                    if (self.y > 1.0 && self.dy > 0.0) || (self.y < -1.0 && self.dy < 0.0) {
                        self.dy = -self.dy;

                        // save overshot amount to hit correct frequency.
                        if self.y > 1.0 {
                            self.y = 2.0 - self.y;
                        } else {
                            self.y = -self.y - 2.0;
                        }
                    }
                    // calculate current phase
                    if phase_adjustment {
                        phase_prog = if self.dy > 0.0 {
                            (self.y + 1.0) / 4.0
                        } else {
                            // shifts range from [1,-1] to [2,4] and then to [0.5,1]
                            (-self.y + 3.0) / 4.0
                        };
                    }
                }
            }

            // calculates phase error to adjust the frequency in the next sample
            if phase_adjustment {
                let mut phase_err = (((progress + wave.offset.unwrap() as f64)
                    * wave.frequency as f64)
                    % 1.0) as f32
                    - phase_prog;

                if phase_err < 0.5 {
                    phase_err += 1.0;
                }
                if phase_err > 0.5 {
                    phase_err -= 1.0;
                }

                self.last_phase_err = phase_err;
            } else {
                self.last_phase_err = 0.0;
            }

            self.effective_amplify =
                self.effective_amplify * 0.99 + group_amplify * wave.amplify * 0.01;

            self.y * self.effective_amplify
        }
    }
}
