use crate::{
    game::{tiles::CubeCoordinate, GameSimSet, Map},
    AppState,
};
use bevy::prelude::*;
use serde::{Deserialize, Serialize};

pub struct RgbTilePlugin;

impl Plugin for RgbTilePlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.add_event::<ChangeRgbTileColor>();
        app.add_event::<RgbTileChangedEvent>();
        app.add_systems(
            Update,
            update_tile_data
                .run_if(in_state(AppState::InGame))
                .in_set(GameSimSet::Prepare),
        );
    }
}

#[derive(Component, Clone, Debug, Serialize, Deserialize)]
/// Simple raw RGB color data
/// stores values between 0 and 255
pub struct RgbTileData {
    pub r: u8,
    pub g: u8,
    pub b: u8,
}

impl Default for RgbTileData {
    fn default() -> Self {
        Self {
            r: 255,
            g: 255,
            b: 255,
        }
    }
}

#[cfg(feature = "graphics")]
impl From<&RgbTileData> for Color {
    fn from(dt: &RgbTileData) -> Self {
        Color::srgb_u8(dt.r, dt.g, dt.b)
    }
}

#[derive(Debug, Serialize, Deserialize, Clone, Event)]
pub struct ChangeRgbTileColor {
    pub coord: CubeCoordinate,
    pub rgb: RgbTileData,
}

#[derive(Debug, Serialize, Deserialize, Clone, Event)]
pub struct RgbTileChangedEvent(pub CubeCoordinate);

/// update the tile components from the ChangeRgbTileColor event
pub fn update_tile_data(
    mut tiles: Query<(Entity, &mut RgbTileData)>,
    map: Res<Map>,
    mut events: EventReader<ChangeRgbTileColor>,
    mut succ: EventWriter<RgbTileChangedEvent>,
) {
    for ChangeRgbTileColor { coord, rgb } in events.read() {
        if let Some(entity) = map.get(*coord) {
            if let Ok(mut data) = tiles.get_mut(entity) {
                *data.1 = rgb.clone();
                succ.send(RgbTileChangedEvent(*coord));
            }
        }
    }
}
