use crate::{
    game::{tiles::CubeCoordinate, GameSimSet, Map},
    AppState,
};
use bevy::prelude::*;
use bevy_enum_filter::Enum;
use serde::{Deserialize, Serialize};

use super::{tile_type_filters, TileType};

const MAX_TEXT_LINES: usize = 10;
const MAX_TEXT_LINE_LENGTH: usize = 26;

pub struct ChameleonTilePlugin;

impl Plugin for ChameleonTilePlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.add_event::<ChangeChameleonTile>();
        app.add_event::<ChameleonTileChangedEvent>();
        app.add_systems(
            Update,
            update_tile_data
                .run_if(in_state(AppState::InGame))
                .in_set(GameSimSet::Prepare),
        );
    }
}

#[derive(Debug, Clone, Component, Serialize, Deserialize)]
pub struct ChameleonTileData {
    pub model: TileType,
    pub text: String,
}

#[derive(Debug, Clone, Default, Component)]
pub struct ChameleonTileUpdate {
    pub model: bool,
    pub text: bool,
}

impl Default for ChameleonTileData {
    fn default() -> Self {
        Self {
            model: TileType::Wheat,
            text: String::new(),
        }
    }
}

#[derive(Debug, Serialize, Deserialize, Clone, Event)]
pub struct ChangeChameleonTile {
    pub coord: CubeCoordinate,
    pub data: ChameleonTileData,
}

/// Gets send on `ChangeChameleonTile` success
#[derive(Debug, Serialize, Deserialize, Clone, Event)]
pub struct ChameleonTileChangedEvent(pub CubeCoordinate);

/// update the tile components from the ChangeChameleonTile event
pub fn update_tile_data(
    mut tiles: Query<
        (&mut ChameleonTileData, &mut ChameleonTileUpdate),
        With<Enum!(TileType::Chameleon)>,
    >,
    map: Res<Map>,
    mut events: EventReader<ChangeChameleonTile>,
    mut event_succ: EventWriter<ChameleonTileChangedEvent>,
) {
    for ChangeChameleonTile { coord, data } in events.read() {
        let Some(entity) = map.get(*coord) else {
            continue;
        };

        let Ok((mut curr_data, mut update)) = tiles.get_mut(entity) else {
            continue;
        };
        if curr_data.text != data.text {
            curr_data.text = data
                .text
                .split("\n")
                .take(MAX_TEXT_LINES)
                .map(|text| text.chars().take(MAX_TEXT_LINE_LENGTH).collect::<String>())
                .collect::<Vec<_>>()
                .join("\n");

            update.text = true;
        }
        if curr_data.model != data.model {
            curr_data.model = data.model;
            update.model = true;
        }
        if update.text || update.model {
            event_succ.send(ChameleonTileChangedEvent(*coord));
        }
    }
}
