pub mod beehive;
pub mod doublehouse;
pub mod forest;
pub mod house;
pub mod marketplace;
pub mod moai;
pub mod quarry;
pub mod wheat;
pub mod windmill;

use std::time::Duration;

use bevy::prelude::*;
use doublehouse::calc_doublehouse_tile;
use marketplace::apply_changes_from_trade_ratio;
use marketplace::calc_marketplace_tile;
use marketplace::re_collect_resource_rates;
use marketplace::track_market_economy;
use marketplace::{ResourceAmount, SellKeepRatio};
use serde::Deserialize;
use serde::Serialize;

use self::{
    beehive::calc_beehive_tile, forest::calc_forest_tile, house::calc_smallhouse_tile,
    moai::calc_moai_tile, quarry::calc_quarry_tile, wheat::calc_wheat_tile,
    windmill::calc_windmill_tile,
};

use super::gamemodes::zen::ZenGameState;
use super::map::TileSaveData;
use super::metrics::TargetMetrics;
use super::round::RoundCounter;
use super::round::RoundTimer;
#[cfg(feature = "graphics")]
use super::visuals::debug_text::debug_text_tick;
use super::{
    build_area::setup_build_area,
    controls::PlacerSet,
    gamemodes::{
        campaign::levels::{
            challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
        },
        challenge::ChallengeGameState,
    },
    hand::{generate_hand, reset_deck},
    metrics::{reset_metrics, reset_target_metrics, Metrics, MetricsRate},
    reset_game_state,
    rewards::reset_rewards_timer,
    round::{continue_timer, reset_round_resources, update_redraw_cost},
    tiles::TileType,
    time::Game,
    time::{reset_game_time, unpause_game_time},
    GamePauseState, GameSimSet,
};

use crate::coordinates::CubeCoordinate;
use crate::prelude::*;

pub const RESOURCE_TICK_INTERVAL: Duration = Duration::from_millis(1000);

/// Send this Event, when the resource rate should be recalculated.
/// Position is used for visuals only. None means do not care about the position.
/// If position is needed remove the Option.
#[derive(Event, Debug, Clone)]
pub struct TilePlacedEvent(pub TileSaveData);

/// Send this Event, when the resource rate should be recalculated.
#[derive(Event, Debug, Clone)]
pub struct TileReclaimedEvent(pub CubeCoordinate);

#[derive(SystemSet, Debug, Clone, PartialEq, Eq, Hash)]
pub enum ResourceCalcSet {
    // calculation pipeline
    Prepare,
    BoosterCalculate,
    Calculate,
    MarketCalculate,
    CollectRates,
}

/// Struct to hold Booster-tile boost value.
#[derive(Default, Component, Deref, DerefMut)]
pub struct BoosterTile(
    /// boost value of the booster tile
    f32,
);

/// Holds all relevant data of a single marketplace
#[derive(Component, Default, Clone, Debug, Serialize, Deserialize)]
pub struct MarketInfo {
    /// How many of the collected Resources should be sold for cash money (relative value).
    pub trade: SellKeepRatio,
    /// How many Resources the market collects in its COLLECT_RADIUS (absolute value).
    #[serde(skip)]
    pub available: ResourceAmount,
    /// At which exchange rate the resources are being sold. Used as a multiplier on sold resources.
    ///
    /// e.g. if you are selling 5 food with a trade efficiency of 0.5 you generate 5 * 0.5 = 2.5 money
    ///
    /// This value is effected by how many Houses (small and double) are in BOOST_RADIUS
    #[serde(skip)]
    pub trade_efficiency: f32,
    /// lifetime economy. Saves metrics delta for this market place.
    pub economy: MetricsRate,
}

/// Use this event to configure a market place, by coordinate without accessing it directly.
/// Keep in mind there is still the `ChangedTradeRatio` Event.
/// This Event implies a `ChangedTradeRatio` Event. So only this one has to be send.
#[derive(Debug, Serialize, Deserialize, Clone, Event)]
#[allow(dead_code)]
pub struct ConfigureMarketReq {
    pub coord: CubeCoordinate,
    pub ratio: SellKeepRatio,
}

/// Send on success of `ConfigureMarket`
#[derive(Debug, Serialize, Deserialize, Clone, Event)]
#[allow(dead_code)]
pub struct MarketConfiguredEvent {
    pub coord: CubeCoordinate,
    pub ratio: SellKeepRatio,
}

impl From<MarketConfiguredEvent> for ConfigureMarketReq {
    fn from(value: MarketConfiguredEvent) -> Self {
        ConfigureMarketReq {
            coord: value.coord,
            ratio: value.ratio,
        }
    }
}

#[derive(Debug, Clone, Copy)]
/// Type of relevant tile.
pub enum RelKind {
    Primary,
    Secondary,
}

/// Saves relevant tiles of tiles for resoure production.
#[derive(Debug, Clone, Component, Default)]
pub struct RelevantTileMarker {
    pub active: bool,
    pub coords: Vec<(CubeCoordinate, RelKind)>,
}

impl RelevantTileMarker {
    pub fn new(active: bool) -> Self {
        Self {
            active,
            coords: Vec::default(),
        }
    }
    pub fn reset(&mut self) {
        self.coords.clear();
    }
    pub fn add(&mut self, coord: CubeCoordinate, kind: RelKind) {
        if self.active {
            self.coords.push((coord, kind));
        }
    }
    pub fn replace(&mut self, coord: Vec<(CubeCoordinate, RelKind)>) {
        if self.active {
            self.coords = coord;
        }
    }
}

#[derive(Debug, Resource, Deref, DerefMut)]
pub struct ResourceTickTimer(pub Timer);

impl Default for ResourceTickTimer {
    fn default() -> Self {
        Self(Timer::from_seconds(
            RESOURCE_TICK_INTERVAL.as_secs_f32(),
            TimerMode::Repeating,
        ))
    }
}

#[derive(Debug, Event)]
pub struct ResourceTickEvent;

#[derive(Debug, Component)]
pub struct DefaultInfoMarker;

macro_rules! define_resource_plugin {
    ($({
        state: $state:expr$(,)?
    }),*) => {
        pub struct ResourceManagerPlugin;
        impl Plugin for ResourceManagerPlugin {
            fn build(&self, app: &mut bevy::prelude::App) {
                app.add_event::<TilePlacedEvent>();
                app.add_event::<TileReclaimedEvent>();
                app.add_event::<ConfigureMarketReq>();
                app.add_event::<MarketConfiguredEvent>();
                app.init_resource::<Events<ResourceTickEvent>>();
                app.init_resource::<ResourceTickTimer>();

                app.add_systems(
                    Update,
                    tick_timer
                        .before(GameSimSet::Simulation)
                        .run_if(in_state(GamePauseState::Running))
                        .run_if(never()$(.or(in_state($state)))*),
                );

                // impl custom event cleanup strategy
                // resource tick always ends on the end of frame
                app.add_systems(PostUpdate, clean_up_tick);

                app.configure_sets(
                    Update,
                    (
                        ResourceCalcSet::Prepare,
                        ResourceCalcSet::BoosterCalculate,
                        ResourceCalcSet::Calculate,
                        ResourceCalcSet::MarketCalculate,
                        ResourceCalcSet::CollectRates,
                    )
                        .chain()
                        .in_set(GameSimSet::Simulation)
                        .run_if(in_state(GamePauseState::Running))
                        .run_if(on_event::<TilePlacedEvent>.or(on_event::<TileReclaimedEvent>))
                        .run_if(never()$(.or(in_state($state)))*)
                );

                app.add_systems(
                    Update,
                    (update_resources, track_market_economy)
                        .in_set(GameSimSet::Simulation)
                        .after(ResourceCalcSet::CollectRates)
                        .run_if(on_event::<ResourceTickEvent>)
                        .run_if(in_state(GamePauseState::Running))
                        .run_if(never()$(.or(in_state($state)))*),
                );

                app.add_systems(
                    Update,
                    (
                        apply_changes_from_trade_ratio,
                        re_collect_resource_rates,
                    )
                        .chain()
                        .in_set(GameSimSet::Prepare)
                        .after(PlacerSet::Placer)
                        .run_if(in_state(GamePauseState::Running))
                        .run_if(never()$(.or(in_state($state)))*)
                );

                #[cfg(feature = "graphics")]
                app.add_systems(
                    Update,
                    debug_text_tick.in_set(ResourceCalcSet::Prepare),
                );
                app.add_systems(
                    Update,
                    calc_moai_tile.in_set(ResourceCalcSet::BoosterCalculate),
                );
                app.add_systems(
                    Update,
                    calc_forest_tile.in_set(ResourceCalcSet::Calculate),
                );
                app.add_systems(
                    Update,
                    calc_quarry_tile.in_set(ResourceCalcSet::Calculate),
                );
                app.add_systems(
                    Update,
                    calc_wheat_tile.in_set(ResourceCalcSet::Calculate),
                );
                app.add_systems(
                    Update,
                    calc_windmill_tile.in_set(ResourceCalcSet::Calculate),
                );
                app.add_systems(
                    Update,
                    calc_smallhouse_tile.in_set(ResourceCalcSet::Calculate),
                );
                app.add_systems(
                    Update,
                    calc_doublehouse_tile.in_set(ResourceCalcSet::Calculate),
                );
                app.add_systems(
                    Update,
                    calc_beehive_tile.in_set(ResourceCalcSet::Calculate),
                );
                app.add_systems(
                    Update,
                    calc_marketplace_tile.in_set(ResourceCalcSet::MarketCalculate),
                );

                app.add_systems(
                    Update,
                    collect_resource_rates.in_set(ResourceCalcSet::CollectRates),
                );
                app.add_systems(
                    Update,
                    print_challenge_tick
                        .run_if(never()$(.or(still_in_state($state)))*)
                        .run_if(on_event::<ResourceTickEvent>)
                        .in_set(GameSimSet::NextAuto),
                );
                $(
                    app.add_systems(
                        OnEnter($state),
                        print_challenge_tick
                            .after(continue_timer)
                            .after(update_redraw_cost)
                            .after(reset_round_resources)
                            .after(generate_hand)
                            .after(reset_deck)
                            .after(reset_game_state)
                            .after(reset_game_time)
                            .after(reset_metrics)
                            .after(unpause_game_time)
                            .after(setup_build_area)
                            .after(reset_target_metrics)
                            .after(reset_rewards_timer),
                    );
                )*
            }
        }
    };
}

define_resource_plugin![
    {
        state: ChallengeGameState::InRound,
    },
    {
        state: CampaignChallengeState::InRound,
    },
    {
        state: CampaignWettbewerbState::InRound,
    },
    {
        state: ZenGameState::InRound,
    }
];

pub fn tick_timer(
    mut timer: ResMut<ResourceTickTimer>,
    time: Res<Time<Game>>,
    mut event: EventWriter<ResourceTickEvent>,
) {
    timer.tick(time.delta());
    if timer.just_finished() {
        event.send(ResourceTickEvent);
    }
}

pub fn clean_up_tick(mut events: ResMut<Events<ResourceTickEvent>>) {
    events.clear();
}

pub fn update_resources(metrics_rate: Res<MetricsRate>, mut metrics: ResMut<Metrics>) {
    *metrics += &metrics_rate;
    debug!("{:?}", metrics);
}

pub fn collect_resource_rates(
    mut metrics: ResMut<MetricsRate>,
    metrics_rate: Query<&MetricsRate, With<TileType>>,
) {
    *metrics = MetricsRate::default();
    for rate in metrics_rate.iter() {
        *metrics += rate;
    }
}

pub fn print_challenge_tick(
    round_counter: Res<RoundCounter>,
    round_timer: Res<RoundTimer>,
    metrics: Res<Metrics>,
    metrics_target: Res<TargetMetrics>,
) {
    info!(
        "ChallengeTick: round: {} ; time: {:.2} ; metrics: f:{:.2} m:{:.2} g:{:.2} ; goal: {:.2}",
        **round_counter,
        round_timer.remaining_secs(),
        metrics.food,
        metrics.materials,
        metrics.money,
        metrics_target.money
    );
}

pub trait HumanReadableNumber {
    fn as_human_readable(&self) -> String;
}
impl HumanReadableNumber for f32 {
    fn as_human_readable(&self) -> String {
        if self.abs() < f32::EPSILON {
            return "0".to_string();
        }

        // support for negative numbers
        if self < &0.0 {
            return format!("-{}", (-self).as_human_readable());
        }

        let base: f32 = 10.0f32.powi(3);
        let mut start = 10.0f32.powi(15);
        for sym in ["P", "T", "G", "M", "k", "", "m", "µ", "n"] {
            if *self >= start {
                if *self >= start * 10.0 {
                    return format!("{:.0}{}", self / start, sym);
                } else {
                    return format!("{:.1}{}", self / start, sym);
                }
            }
            start /= base;
        }

        // numbers really small
        format!("{:.3}p", *self / start)
    }
}
