use std::collections::HashSet;

use bevy::ecs::prelude::*;
use bevy_enum_filter::Enum;

use super::{BoosterTile, RelKind, RelevantTileMarker};
use crate::{
    coordinates::CubeCoordinate,
    game::{
        map::Map,
        tiles::{tile_type_filters, TileType},
    },
};

#[cfg(feature = "graphics")]
use crate::game::visuals::debug_text::DebugText;

// ********************** Tweak this **********************
/// Boost Range of Moai Tile. Tweak this for game balancing
pub const BOOSTRANGE_MOAI: usize = 3;
/// Counts number of unique tiles in this range around the moai. Tweak this for game balancing
const DIVERSITY_RANGE: usize = 2;
/// How much the boost power increases per diversity. Tweak this for game balancing
const BOOSTPOWER_PER_DIVERSITY: f32 = 0.5;
/// Moai will have a default boost value defined here. Tweak this for game balancing
const BASE_BOOST: f32 = 1.;
/// Defines how diverse neighbors have to be for boost to kick in. Tweak this for game balancing
const DIVERSITY_THRESHOLD: usize = 4;
// ********************************************************

pub fn calc_moai_tile(
    map: Res<Map>,
    #[cfg(feature = "graphics")] mut debug_text: Query<&mut DebugText, With<Enum!(TileType::Moai)>>,
    mut moai: Query<
        (
            Entity,
            &CubeCoordinate,
            &mut BoosterTile,
            &mut RelevantTileMarker,
        ),
        With<Enum!(TileType::Moai)>,
    >,
    tiles: Query<&TileType>,
) {
    for (_entity, &i, mut boost, mut relevant) in moai.iter_mut() {
        relevant.reset();
        // count to get amount of unique tile types in diversity_range (trivial)
        let diversity_count = map
            .get_area_with_coords(i, DIVERSITY_RANGE) // get all tiles in range
            .filter_map(|(entity, coord)| tiles.get(entity).ok().map(|tile| (tile, coord))) // get their tile type
            .fold(HashSet::new(), |mut set, (&next_tile, coord)| {
                // mark as relevant
                relevant.add(coord, RelKind::Primary);
                // put tile type in set (if already in set nothing happens)
                set.insert(next_tile);
                set
            })
            .len(); // number of elements in set
                    // saturating sub: max(a-b, 0)
        let boost_value = BASE_BOOST
            + (BOOSTPOWER_PER_DIVERSITY
                * (diversity_count.saturating_sub(DIVERSITY_THRESHOLD)) as f32);

        // save boost value
        **boost = boost_value;

        // display debug text
        #[cfg(feature = "graphics")]
        if let Ok(mut t) = debug_text.get_mut(_entity) {
            t.add_section(
                "Moai Tile:",
                format!("Boost: {:.1}\nRange: {}", boost_value, BOOSTRANGE_MOAI),
            );
        }
    }
}
