use bevy::prelude::*;
use bevy_enum_filter::Enum;

use crate::{
    coordinates::CubeCoordinate,
    game::{map::Map, metrics::MetricsRate, tiles::tile_type_filters},
};

#[cfg(feature = "graphics")]
use crate::game::visuals::debug_text::DebugText;

use super::{RelKind, RelevantTileMarker};

/// Base Production of forest tiles. Tweak this for game balancing
pub const BASE_PRODUCTION_RATE: f32 = 0.25;
/// Additive bonus per adjacent forest tile. Tweak this for game balancing
pub const BONUS_PER_NEIGHBOR: f32 = 0.5;

pub fn calc_forest_tile(
    map: Res<Map>,
    #[cfg(feature = "graphics")] mut debug_text: Query<
        &mut DebugText,
        With<Enum!(TileType::Forest)>,
    >,
    forest: Query<(Entity, &CubeCoordinate), With<Enum!(TileType::Forest)>>,
    mut forest_rates: Query<
        (&mut MetricsRate, &mut RelevantTileMarker),
        With<Enum!(TileType::Forest)>,
    >,
) {
    // iterate all forest tiles
    for (entity, coord) in forest.iter() {
        let mut neighbors: Vec<(CubeCoordinate, RelKind)> = Vec::new();
        // get neighbors of forest tiles
        for (entity, coord) in map.get_neighbors_with_coords(*coord) {
            // check if neighbor is also a forest tile
            if forest.contains(entity) {
                neighbors.push((coord, RelKind::Primary));
            }
        }

        let production_rate = (neighbors.len() as f32 * BONUS_PER_NEIGHBOR) + BASE_PRODUCTION_RATE;

        #[cfg(feature = "graphics")]
        if let Ok(mut t) = debug_text.get_mut(entity) {
            t.add_section(
                "Forest Tile:",
                format!(
                    "Neighbor_count: {} \nproduces: {:.2}",
                    neighbors.len(),
                    production_rate
                ),
            );
        }
        if let Ok((mut rate, mut relevant)) = forest_rates.get_mut(entity) {
            relevant.replace(neighbors);
            *rate = MetricsRate {
                food: 0.0,
                materials: production_rate,
                money: 0.0,
            };
        }
    }
}
