use bevy::prelude::*;
use bevy_enum_filter::Enum;

use crate::{
    coordinates::{lerp, CubeCoordinate},
    game::{
        map::Map, metrics::MetricsRate, resources::moai::BOOSTRANGE_MOAI, tiles::tile_type_filters,
    },
};

#[cfg(feature = "graphics")]
use crate::game::visuals::debug_text::DebugText;

use super::{BoosterTile, RelKind, RelevantTileMarker};

/// Maximum of the single saw tooth function at PRODUCTION_RATE_OPTIMUM
const PRODUCTION_RATE_MAX: f32 = 5.0;
/// Location of optimum of the single saw tooth function.
const PRODUCTION_RATE_OPTIMUM: f32 = 3.0;
/// The Minimum of the single saw tooth function at x = 0 and x = 6.
const PRODUCTION_RATE_MIN: f32 = 0.5;
/// DO NOT CHANGE. Maximum possible neighbors.
const MAX_NEIGHBORS: f32 = 6.;

pub fn calc_doublehouse_tile(
    map: Res<Map>,
    #[cfg(feature = "graphics")] mut debug_text: Query<
        &mut DebugText,
        With<Enum!(TileType::DoubleHouse)>,
    >,
    smallhouses: Query<&CubeCoordinate, With<Enum!(TileType::SmallHouse)>>,
    doublehouses: Query<(Entity, &CubeCoordinate), With<Enum!(TileType::DoubleHouse)>>,
    mut doublehouse_data: Query<
        (&mut MetricsRate, &mut RelevantTileMarker),
        With<Enum!(TileType::DoubleHouse)>,
    >,
    moai: Query<(&CubeCoordinate, &BoosterTile), With<BoosterTile>>,
) {
    // iterate all DoubleHouse tiles
    for (entity, coord) in doublehouses.iter() {
        let mut neighbors: Vec<(CubeCoordinate, RelKind)> = Vec::new();

        let mut small_neighbors = 0;
        // get smallhouse neighbors
        for (entity, coord) in map.get_neighbors_with_coords(*coord) {
            if smallhouses.contains(entity) {
                neighbors.push((coord, RelKind::Primary));
                small_neighbors += 1;
            }
        }
        let mut double_neighbors = 0;
        // get doublehouse neighbors
        for (entity, coord) in map.get_neighbors_with_coords(*coord) {
            if doublehouses.contains(entity) {
                neighbors.push((coord, RelKind::Primary));
                double_neighbors += 1;
            }
        }
        let house_neighbors: f32 = (double_neighbors + small_neighbors) as f32;

        // Saw-plate (tooth) function with maximum at PRODUCTION_RATE_OPTIMUM and minimum at 0 and 6.
        let mut production_rate = if house_neighbors <= PRODUCTION_RATE_OPTIMUM {
            lerp(
                PRODUCTION_RATE_MIN,
                PRODUCTION_RATE_MAX,
                house_neighbors / PRODUCTION_RATE_OPTIMUM,
            )
        } else {
            lerp(
                PRODUCTION_RATE_MAX,
                PRODUCTION_RATE_MIN,
                (house_neighbors - PRODUCTION_RATE_OPTIMUM)
                    / (MAX_NEIGHBORS - PRODUCTION_RATE_OPTIMUM),
            )
        };

        // find moai tile with highest boost value in range
        let max_booster = moai
            .iter()
            .filter(|(boost_coord, _)| coord.distance(**boost_coord) <= BOOSTRANGE_MOAI)
            .fold((1.0, None), |(max, coord), (new_coord, boost)| {
                if max > **boost {
                    (max, coord)
                } else {
                    (**boost, Some(new_coord))
                }
            });

        production_rate *= max_booster.0;

        if let Ok((mut rate, mut relevant)) = doublehouse_data.get_mut(entity) {
            relevant.replace(neighbors);
            if let Some(coord) = max_booster.1 {
                relevant.add(*coord, RelKind::Secondary);
            }
            *rate = MetricsRate {
                food: 0.0,
                materials: 0.0,
                money: production_rate,
            };
        }

        #[cfg(feature = "graphics")]
        if let Ok(mut t) = debug_text.get_mut(entity) {
            t.add_section(
                "DoubleHouse Tile:",
                format!("produces: {:.1}", production_rate),
            );
        }
    }
}
