use bevy::prelude::*;
use serde::{Deserialize, Serialize};

#[cfg(target_family = "wasm")]
use crate::download;
use std::ops::AddAssign;
#[cfg(not(target_family = "wasm"))]
use std::{fs, io::Write};

use crate::GameConfig;

use super::{
    gamemodes::{
        campaign::{levels::CampaignGroupWrapper, CampaignModeConfig},
        challenge::ChallengeModeConfig,
        GameMode,
    },
    metrics::Metrics,
    recorder::Record,
    replayer::{clean_replay, Replay},
    round::RoundCounter,
    GameSimSet, Successful,
};
macro_rules! define_reporter_plugin {
    ($({
        // in which sub game mode to run the record plugin
        gamemode: $gm:expr,
    }),*) => {
        pub struct ReportPlugin;

        impl Plugin for ReportPlugin {
            fn build(&self, app: &mut App) {
                app.init_resource::<Warnings>();
                app.add_event::<Warnings>();
                use crate::prelude::never;
                $(
                    app.add_systems(
                        OnEnter($gm),
                        clean_warnings
                    );
                    app.add_systems(
                        OnExit($gm),
                        save_report
                            .run_if(report_activated)
                            .before(clean_replay),
                    );
                )*
                app.add_systems(
                    Update,
                    collect_warnings
                        .in_set(GameSimSet::AfterSim)
                        .run_if(never()$(.or(in_state($gm)))*)
                );
                // HACK: save on exit.
                // this might not work if exit is too fast.
                // But should work, because it runs in PostUpdate, and is able to receive the exit event in the same tick as send.
                app.add_systems(
                    PostUpdate,
                    save_report
                        .run_if(never()$(.or(in_state($gm)))*)
                        .run_if(on_event::<AppExit>)
                        .run_if(report_activated),
                );
            }
        }
    };
}

define_reporter_plugin![
    {
        gamemode: GameMode::Challenge,
    },
    {
        gamemode: GameMode::Creative,
    },
    {
        gamemode: GameMode::Campaign,
    },
    {
        gamemode: GameMode::Zen,
    }
];

fn report_activated(config: Res<GameConfig>) -> bool {
    config.report_file.is_some()
}

fn clean_warnings(mut warnings: ResMut<Warnings>) {
    *warnings = Warnings::default();
}

fn collect_warnings(mut warnings: ResMut<Warnings>, mut event: EventReader<Warnings>) {
    for warning in event.read() {
        *warnings += warning;
    }
}

#[derive(Debug, Default, Serialize, Deserialize)]
pub struct Report {
    gamemode: GameMode,
    #[serde(default)]
    round: Option<RoundCounter>,
    #[serde(default)]
    record: Option<Record>,
    #[serde(default)]
    pub campaign: Option<CampaignGroupWrapper>,
    #[serde(default)]
    pub challenge: Option<ChallengeModeConfig>,
    #[serde(default)]
    metrics: Option<Metrics>,
    #[serde(default)]
    warnings: Option<Warnings>,
    #[serde(default)]
    success: Successful,
}

impl Report {
    pub fn new(gamemode: GameMode) -> Self {
        Self {
            gamemode,
            ..default()
        }
    }
}

#[derive(Debug, Clone, Default, Serialize, Deserialize, Resource, Event, PartialEq, Eq)]
pub struct Warnings {
    place: usize,
    take: usize,
    reward: usize,
    redraw: usize,
    market: usize,
}

impl Warnings {
    pub fn place() -> Self {
        Self {
            place: 1,
            ..Default::default()
        }
    }
    pub fn take() -> Self {
        Self {
            take: 1,
            ..Default::default()
        }
    }
    pub fn reward() -> Self {
        Self {
            reward: 1,
            ..Default::default()
        }
    }
    pub fn redraw() -> Self {
        Self {
            redraw: 1,
            ..Default::default()
        }
    }
    pub fn market() -> Self {
        Self {
            market: 1,
            ..Default::default()
        }
    }
}

impl AddAssign<&Warnings> for Warnings {
    fn add_assign(&mut self, rhs: &Warnings) {
        self.place += rhs.place;
        self.take += rhs.take;
        self.reward += rhs.reward;
        self.redraw += rhs.redraw;
        self.market += rhs.market;
    }
}

#[allow(clippy::too_many_arguments)]
fn save_report(
    metrics: Res<Metrics>,
    config: Res<GameConfig>,
    round: Res<RoundCounter>,
    challenge_config: Option<Res<ChallengeModeConfig>>,
    campaign_config: Option<Res<CampaignModeConfig>>,
    replay: Option<Res<Replay>>,
    warnings: Res<Warnings>,
    succ: Res<Successful>,
) {
    let mut report = Report::new(config.selected_gamemode.clone());
    if report.gamemode == GameMode::Challenge {
        // save challenge config
        report.challenge = challenge_config.map(|f| f.clone());

        // results
        report.metrics = Some((*metrics).clone());
        report.round = Some(RoundCounter(**round));
    }
    if report.gamemode == GameMode::Campaign {
        // save campaign config
        report.campaign = campaign_config.map(|f| f.level.clone());
    }
    if let Some(replay) = replay {
        report.record = Some(replay.0.clone());
    }
    report.success = succ.clone();
    report.warnings = Some(warnings.clone());
    if let Some(path) = &config.report_file {
        match serde_json::to_string(&report) {
            Ok(data) => {
                #[cfg(not(target_family = "wasm"))]
                match fs::File::create(path).and_then(|mut file| file.write_all(data.as_bytes())) {
                    Ok(_) => {
                        info!("Saved game report to: {}", path);
                    }
                    Err(err) => warn!("Error while saving record: {}", err),
                }

                #[cfg(target_family = "wasm")]
                download(&data, path, "text/json");
            }
            Err(err) => warn!("Error while saving the game record: {}", err),
        };
    }
}
