use bevy::prelude::*;
use serde::{Deserialize, Serialize};

use super::{GameMode, GameModeSubStates};
#[cfg(feature = "graphics")]
use crate::{
    game::time::pause_on_focus_loss,
    ui::in_game::{
        game_export::show_export_window,
        pause::{ui_pause_screen, PauseUiState},
    },
};
use crate::{
    game::{
        hand::{update_deck, Deck, RandomCore},
        round::{RoundCounter, RoundSystemSupport, RoundTimer},
        Successful,
    },
    AppState, DisplayMode,
};

pub struct ZenModePlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for ZenModePlugin {
    fn build(&self, app: &mut App) {
        app.add_sub_state::<ZenGameState>();
        app.init_resource::<ZenModeConfig>();

        use ZenGameState::*;

        app.add_systems(Update, refill_deck.run_if(in_state(InRound)));

        #[cfg(feature = "graphics")]
        if let DisplayMode::Graphic = self.display_mode {
            use GameMode::Zen;

            app.add_systems(
                Update,
                ui_pause_screen::<true>
                    .run_if(in_state(Zen))
                    .run_if(in_state(PauseUiState::Shown))
                    .before(show_export_window),
            );
            app.add_systems(Update, pause_on_focus_loss.run_if(in_state(InRound)));
        }
    }
}

/// refills the deck, whenever the round timer finishes
/// to simulate the challenge-mode refill cycle
/// NOTE: only refills the deck if the round hasn't just finished
fn refill_deck(
    next: Res<NextState<ZenGameState>>,
    mut round_timer: ResMut<RoundTimer>,
    deck: ResMut<Deck>,
    round: Res<RoundCounter>,
) {
    if let NextState::Unchanged = *next {
        // we are still in the round
        if round_timer.just_finished() {
            // virtual round is over
            round_timer.reset();
            // refill hand
            update_deck(deck, round);
            debug!("Refilled deck");
        }
    }
}

#[derive(Debug, Clone, Copy, Default, PartialEq, Eq, Hash, SubStates)]
#[source(GameMode = GameMode::Zen)]
pub enum ZenGameState {
    /// the player is currently in a round
    /// and can place tiles and redraw cards
    #[default]
    InRound,
    /// the player is between two rounds
    BetweenRounds,
}

impl GameModeSubStates for ZenGameState {}

impl RoundSystemSupport for ZenGameState {
    fn get_gameover_state() -> Self {
        unreachable!("The game cannot be lost in zen mode")
    }
    fn get_phaseend_state() -> Self {
        Self::BetweenRounds
    }
    fn get_inround_state() -> Self {
        Self::InRound
    }
    fn get_spectate_state() -> Option<Self> {
        None
    }
}

#[derive(Debug, Clone, Default, Resource, Serialize, Deserialize)]
pub struct ZenModeConfig {
    pub seed: String,
}

/// one-shot system to start a new game in challenge mode
/// loads gamemode and configuration data from the GameConfig resource
/// ```ignore
/// world.run_system_once(init_challenge_mode)
/// ```
pub fn init_zen_mode(
    config: Res<ZenModeConfig>,
    mut seed: ResMut<RandomCore>,
    mut app_state: ResMut<NextState<AppState>>,
    mut gamemode: ResMut<NextState<GameMode>>,
    mut succ: ResMut<Successful>,
) {
    info!("Starting game with seed: {}", config.seed);
    seed.change_seed(&config.seed);
    app_state.set(AppState::InGame);
    gamemode.set(GameMode::Zen);
    **succ = true;
}
