/// generate all code required for a level
/// has to be called for every level
///
/// every level must have a parent group and a level state name.
///
/// Additionally it has to have a story with at least a start and end segment.
/// The next story segment will be endered,
/// once the goal of the previous segement has been achived
/// Every segment has an `id` (used for translations and the cli)
/// and a `ref` which is the enum variant name used for the state.
/// The start- and mid-segments also have a `goal`, which is as `=>` separated
/// list of run-conditions
macro_rules! define_level {
    (
        $group:ident::$name:ident,
        $(init: $init:ident,)?
        story: [
        {
            id: $start_id:literal
        },
        $(
            {
                ref: $ref:ident,
                id: $id:literal,
                goal: $( $goal:expr )=>+
            },
        )+
        {
            id: $end_id:literal
        }
    ]) => {

        #[derive(SubStates, Default, Debug, Hash, Eq, PartialEq, Clone)]
        #[source($group = $group::$name)]
        pub enum $name {
            #[default]
            Intro,
            $($ref,)+
            Outro
        }

        impl $name {
            fn list() -> Vec<Self> {
                vec![
                    Self::Intro,
                    $(Self::$ref,)+
                    Self::Outro
                ]
            }
        }

        impl crate::game::gamemodes::campaign::traits::Level for $name {
            fn get_id(&self) -> &'static str {
                match self {
                    Self::Intro => $start_id,
                    $(Self::$ref => $id,)+
                    Self::Outro => $end_id
                }
            }
            fn get_level_id(&self) -> &'static str {
                use crate::game::gamemodes::campaign::traits::Group;
                // HACK: we don't have access to the level ID here
                // so we forward the request to the level group
                // that way we only have to define the id once
                $group::$name.get_level_id()
            }
            /// O(n) next implementation,
            /// should be good enough for our usecase,
            /// as it only needs to run once the storysegment goal was reached
            /// additionally there probably wont ever be more than 10/20 segments
            fn next(&self) -> Option<Self> {
                let mut found = false;
                for step in Self::list() {
                    if found {
                        return Some(step);
                    }
                    if *self == step {
                        found = true;
                    }
                }
                return None;
            }
            fn register(app: &mut bevy::prelude::App) {

                use crate::game::gamemodes::campaign::{display_story_segment, step_level_story};
                use crate::game::gamemodes::campaign::CampaignLevelStart;
                use crate::game::GameSimSet;

                app.add_sub_state::<$name>();

                // automatically display the info text when starting the level
                // NOTE: in graphic mode, the user has to click start to transition to the mid section
                app.add_systems(OnEnter(Self::Intro), display_story_segment::<Self>);
                app.add_systems(Update, step_level_story::<Self>
                                .run_if(in_state(Self::Intro))
                                .run_if(on_event::<CampaignLevelStart>));

                // generate code to show the new text for every storyline segment
                // and automatically step into the next section, once the target is reached
                $(
                    app.add_systems(OnEnter(Self::$ref), display_story_segment::<Self>);
                    app.add_systems(Update, step_level_story::<Self>
                                    .run_if(in_state(Self::$ref))
                                    $(.run_if($goal))+
                                    .in_set(GameSimSet::AfterSim));
                )+

                // automatically display the info text once the level is done
                app.add_systems(OnEnter(Self::Outro), display_story_segment::<Self>);

            }
            fn get_progress(&self) -> crate::game::gamemodes::campaign::CampaignLevelProgress {
                use crate::game::gamemodes::campaign::CampaignLevelProgress;
                match self {
                    Self::Intro => CampaignLevelProgress::Intro,
                    $(
                        Self::$ref => CampaignLevelProgress::Running,
                    )+
                    Self::Outro => CampaignLevelProgress::Outro,
                }
            }
            $(
                fn init(world: &mut World) {
                    $init(world)
                }
            )?
        }
    };
}

/// Generates the code for a central group database
/// and all the grouping states
/// should only be called once
///
/// Every group has an `id`, which is used for translations and the cli
/// and a `ref` used as the substate variant.
/// a group also consist of one or more levels.
/// Additionally each group can also have a custom `init` function,
/// which is called by the campaign init function, when a level is started
///
/// Each level has a `lv_id`, used to identify the level (translations and cli)
/// and a `lv_ref` used as a substate variant name,
/// should also be the state name of the level
/// (see `define_levels`)
///
/// `CampaignGroups` provides `register` which can be used to register all substates
/// (including the groups and levels) to the bevy app
macro_rules! campaign_groups {
    ({
        $($id:literal: {
            ref: $ref:ident,
            $(init: $init:ident,)?
            levels: {
                $($lv_id:literal: $lv_ref:ident),+ $(,)?
            }
        }),+ $(,)?
    }) => (

        use crate::game::gamemodes::GameMode;
        use serde::{Serialize, Deserialize};

        /// central campaign group states
        /// the variants represent the group `ref`s
        #[derive(SubStates, Default, Debug, Eq, Hash, PartialEq, Clone)]
        #[source(GameMode = GameMode::Campaign)]
        pub enum CampaignGroups {
            #[default]
            $($ref,)+
        }
        impl CampaignGroups {
            /// shorthand to register all known groups and their levels
            pub fn register(app: &mut bevy::prelude::App) {
                use crate::game::gamemodes::campaign::traits::Group;
                app.add_sub_state::<Self>();
                $($ref::register(app);)+
            }
            /// returns the id of the campaign level group
            pub fn get_id(&self) -> &'static str {
                match self {
                    $(
                        Self::$ref => $id,
                    )+
                }
            }
        }

        /// helper enum to allow selecting the level inside a group
        #[derive(Clone, Debug, PartialEq, Serialize, Deserialize)]
        pub enum CampaignGroupWrapper {
            $($ref($ref),)+
        }

        use crate::game::gamemodes::campaign::CampaignGroupParserError;

        impl CampaignGroupWrapper {
            /// Attempt to parse the value from the level and group
            pub fn from_ids(group: &str, level: &str) -> Result<Self, CampaignGroupParserError> {
                match group {
                    $(
                        $id => match level {
                            $(
                                $lv_id => Ok(Self::$ref($ref::$lv_ref)),
                            )+
                            _ => Err(CampaignGroupParserError::InvalidLevel)
                        },
                    )+
                    _ => Err(CampaignGroupParserError::InvalidGroup)
                }
            }
            /// Attempts to construct a value using the group id
            /// using the group's default level as the selected level
            pub fn from_group_default(group: &str) -> Option<Self> {
                match group {
                    $(
                        $id => Some(Self::$ref($ref::default())),
                    )+
                        _ => None
                }
            }

            /// returns the id of the value group
            pub fn get_group_id(&self) -> &'static str {
                match self {
                    $(
                        Self::$ref(_) => $id,
                    )+
                }
            }
            /// returns the id of the level of the selected group
            pub fn get_level_id(&self) -> &'static str {
                match self {
                    $(
                        Self::$ref(lv) => match lv {
                            $(
                                $ref::$lv_ref => $lv_id,
                            )+
                        },
                    )+
                }
            }

            /// returns a list of all known group ids
            pub fn list_ids() -> Vec<&'static str> {
                vec![
                    $(
                        $id,
                    )+
                ]
            }
            /// returns a list of all known level ids in a given group
            ///
            /// returns None if the group does not exist
            pub fn list_level_ids(group: &str) -> Option<Vec<&'static str>> {
                match group {
                    $(
                        $id => Some(vec![
                            $($lv_id,)+
                        ]),
                    )+
                        _ => None
                }
            }
        }

        // generate substates for every level group
        $(
            #[derive(SubStates, Default, Debug, Eq, Hash, PartialEq, Clone, Serialize, Deserialize)]
            #[source(CampaignGroups = CampaignGroups::$ref)]
            pub enum $ref {
                #[default]
                $($lv_ref,)+
            }
            impl crate::game::gamemodes::campaign::traits::Group for $ref {
                fn get_id(&self) -> &'static str {
                    return $id;
                }
                fn register(app: &mut bevy::prelude::App) {
                    app.add_sub_state::<Self>();
                    use crate::game::gamemodes::campaign::traits::Level;
                    $($lv_ref::register(app);)+
                }
                fn get_level_id(&self) -> &'static str {
                    // NOTE: cannot be done in the level macro,
                    // as it does not know the level id
                    match self {
                        $(
                            Self::$lv_ref => $lv_id,
                        )+
                    }
                }
                $(
                    fn init(world: &mut World) {
                        $init(world)
                    }
                )?
            }
        )*

        /// one-shot system to start a new game in campaign mode
        /// loads gamemode and configuration data from the GameConfig resource
        /// ```ignore
        /// world.run_system_once(init_campaign_mode)
        /// ```
        pub fn init_campaign_mode(
            world: &mut World
        ) {
            {
                use crate::AppState;
                use crate::game::gamemodes::GameMode;
                use crate::game::Successful;
                // set base states
                let mut app_state = world.get_resource_mut::<NextState<AppState>>().expect("The app won't even launch without the AppState");
                app_state.set(AppState::InGame);
                let mut gamemode = world.get_resource_mut::<NextState<GameMode>>().expect("The Gamemode resource is added by the GameMode plugin");
                gamemode.set(GameMode::Campaign);
                let mut succ = world.get_resource_mut::<Successful>().expect("Missing resource Successful. Should be added always.");
                **succ = false;
            }
            use crate::game::gamemodes::campaign::CampaignModeConfig;
            // detect groups and levels
            let level = world.get_resource::<CampaignModeConfig>().expect("Resource is added by the Campaign plugin").level.clone();
            match level {
                $(
                    CampaignGroupWrapper::$ref(group) => {
                        let mut groups = world.get_resource_mut::<NextState<CampaignGroups>>().expect("The CampainGroups resource is added by this macro");
                        groups.set(CampaignGroups::$ref);
                        match group {
                            $(
                                $ref::$lv_ref => {
                                    let mut group = world.get_resource_mut::<NextState<$ref>>().expect("The level resource is added by this macro");
                                    group.set($ref::$lv_ref);
                                    {
                                        use crate::DisplayMode;
                                        use crate::game::gamemodes::campaign::traits::Level;
                                        let display_mode = world.get_resource::<State<DisplayMode>>().expect("The DisplayMode is set on app startup");
                                        use crate::game::gamemodes::campaign::CampaignLevelProgress;
                                        if display_mode.get() == &DisplayMode::Headless {
                                            // skip intro in headless mode
                                            let mut lv_segment = world.get_resource_mut::<NextState<$lv_ref>>().expect("The register method also registers the level states");
                                            lv_segment.set($lv_ref::Intro.next().expect("Every level has at least three segments, so it has to have a first mid segment"));
                                            // reset level progress, but skip the intro segment
                                            // as we immedialy start the level
                                            let mut progress = world.get_resource_mut::<NextState<CampaignLevelProgress>>().expect("The CampaignLevelProgress resource is added by the CampaignModePlugin");
                                            progress.set(CampaignLevelProgress::Running);
                                        } else {
                                            // if we don't skip the intro,
                                            // we have to pause the game, as the IntroScreen requires the game to be paused
                                            // to prevent the time from continuing
                                            use crate::game::GamePauseState;
                                            let mut pause_state = world.get_resource_mut::<NextState<GamePauseState>>().expect("The GamePauseState is registered by the GamePlugin");
                                            pause_state.set(GamePauseState::Paused);
                                            // reset level progress
                                            let mut progress = world.get_resource_mut::<NextState<CampaignLevelProgress>>().expect("The CampaignLevelProgress resource is added by the CampaignModePlugin");
                                            progress.set(CampaignLevelProgress::Intro);
                                        }
                                    }
                                    use crate::game::gamemodes::campaign::traits::Group;
                                    $ref::init(world);
                                    use crate::game::gamemodes::campaign::traits::Level;
                                    $lv_ref::init(world);
                                },
                            )+
                        }
                    },
                )+
            }
        }
    );
}
