use crate::{
    coordinates::CubeCoordinate,
    game::{
        bot::challenge_bot::simulate_bot,
        build_area::BuildArea,
        gamemodes::campaign::levels::CampaignWettbewerb,
        hand::{generate_hand, Hand, RandomCore},
        map::Map,
        placement_reclaim::TileLoaded,
        tiles::{AddTile, RemoveTile, TileType},
    },
    DisplayMode,
};
use bevy::prelude::*;

use super::CampaignWettbewerbState;

pub struct CampaignWettbewerbQuest4Plugin;
impl Plugin for CampaignWettbewerbQuest4Plugin {
    fn build(&self, app: &mut App) {
        app.add_systems(
            OnEnter(CampaignWettbewerbState::InRound),
            (setup_hand, quest4_setup)
                .run_if(in_state(CampaignWettbewerb::Quest4))
                .after(generate_hand)
                .before(simulate_bot),
        );
    }
}

fn quest4_setup(
    mut commands: Commands,
    display_mode: Res<State<DisplayMode>>,
    mut build_area: Query<&mut BuildArea>,
    placed: Query<&CubeCoordinate, With<TileType>>,
) {
    for coord in placed.iter() {
        commands.queue(RemoveTile::new(*coord));
    }
    let map = [
        (CubeCoordinate::new(0, 0, 0), TileType::Forest),
        (CubeCoordinate::new(0, 1, -1), TileType::Forest),
        (CubeCoordinate::new(-4, 3, 1), TileType::Forest),
        (CubeCoordinate::new(-3, 1, 2), TileType::Forest),
        (CubeCoordinate::new(-4, 2, 2), TileType::Forest),
        (CubeCoordinate::new(-3, 4, -1), TileType::Forest),
        (CubeCoordinate::new(-4, 4, 0), TileType::Forest),
        (CubeCoordinate::new(-2, 2, 0), TileType::Forest),
        (CubeCoordinate::new(-2, 3, -1), TileType::Forest),
        (CubeCoordinate::new(-3, 2, 1), TileType::Forest),
        (CubeCoordinate::new(-2, 1, 1), TileType::Forest),
        (CubeCoordinate::new(-1, 1, 0), TileType::Forest),
        (CubeCoordinate::new(-1, 2, -1), TileType::Forest),
        (CubeCoordinate::new(-2, -1, 3), TileType::Wheat),
        (CubeCoordinate::new(-1, -2, 3), TileType::Wheat),
        (CubeCoordinate::new(-1, -1, 2), TileType::Wheat),
        (CubeCoordinate::new(-2, 0, 2), TileType::Wheat),
        (CubeCoordinate::new(-1, 0, 1), TileType::Wheat),
        (CubeCoordinate::new(0, -2, 2), TileType::Wheat),
        (CubeCoordinate::new(0, -1, 1), TileType::Wheat),
        (CubeCoordinate::new(1, -2, 1), TileType::Windmill),
        (CubeCoordinate::new(1, -1, 0), TileType::Wheat),
        (CubeCoordinate::new(-3, 0, 3), TileType::Grass),
        (CubeCoordinate::new(-2, -2, 4), TileType::Grass),
        (CubeCoordinate::new(0, -3, 3), TileType::Grass),
        (CubeCoordinate::new(1, -3, 2), TileType::Grass),
        (CubeCoordinate::new(2, -2, 0), TileType::Grass),
        (CubeCoordinate::new(3, -3, 0), TileType::Grass),
        (CubeCoordinate::new(1, 0, -1), TileType::Grass),
        (CubeCoordinate::new(2, 0, -2), TileType::Grass),
        (CubeCoordinate::new(3, -1, -2), TileType::Grass),
        (CubeCoordinate::new(1, 1, -2), TileType::Grass),
        (CubeCoordinate::new(-1, 3, -2), TileType::Grass),
        (CubeCoordinate::new(-4, 5, -1), TileType::Grass),
        (CubeCoordinate::new(-1, -3, 4), TileType::StoneRocks),
        (CubeCoordinate::new(2, -1, -1), TileType::StoneRocks),
        (CubeCoordinate::new(2, -3, 1), TileType::StoneHill),
        (CubeCoordinate::new(3, -2, -1), TileType::StoneHill),
        (CubeCoordinate::new(0, 2, -2), TileType::StoneHill),
        (CubeCoordinate::new(0, 3, -3), TileType::SmallHouse),
        (CubeCoordinate::new(1, 2, -3), TileType::SmallHouse),
        (CubeCoordinate::new(2, 1, -3), TileType::SmallHouse),
        (CubeCoordinate::new(2, 2, -4), TileType::SmallHouse),
    ];
    for (coord, tile) in map {
        commands.spawn(TileLoaded).queue(
            AddTile::new(tile, coord)
                .without_scene_cond(*display_mode.get() == DisplayMode::Headless)
                .with_data(None),
        );
    }
    if let Ok(mut build_area) = build_area.get_single_mut() {
        build_area.radius = 5;
    }
}

fn setup_hand(mut hand: ResMut<Hand>) {
    hand.items = vec![TileType::Wheat, TileType::Forest, TileType::DoubleHouse];
}

fn test_quest4(map: Res<Map>, tiles: Query<&TileType>) -> bool {
    let test = [
        (CubeCoordinate::new(-3, -1, 4), TileType::Wheat),
        (CubeCoordinate::new(-3, 3, 0), TileType::Forest),
        (CubeCoordinate::new(1, 3, -4), TileType::DoubleHouse),
    ];
    test.iter().all(|(coord, tile)| {
        map.get(*coord)
            .and_then(|e| tiles.get(e).ok())
            .map(|t| *t == *tile)
            .unwrap_or_default()
    })
}

fn quest4_init(world: &mut World) {
    let mut rng_core = world
        .get_resource_mut::<RandomCore>()
        .expect("The corresponding plugin is added by the level group plugin");
    rng_core.change_seed("quest4");
}

define_level!(
    CampaignWettbewerb::Quest4,
    init: quest4_init,
    story: [
        {
            id: "intro"
        },
        {
            ref: Main,
            id: "story-main",
            goal: test_quest4
        },
        {
            id: "done"
        }
    ]
);
