use crate::{
    coordinates::CubeCoordinate,
    game::{
        bot::challenge_bot::simulate_bot,
        build_area::BuildArea,
        gamemodes::campaign::levels::CampaignWettbewerb,
        hand::{generate_hand, Hand},
        map::Map,
        placement_reclaim::TileLoaded,
        round::{process_manual_redraw, Redraw},
        tiles::{tile_type_filters, AddTile, RemoveTile, TileType},
        GameSimSet,
    },
    DisplayMode,
};
use bevy::prelude::*;
use bevy_enum_filter::Enum;

use super::CampaignWettbewerbState;

pub struct CampaignWettbewerbQuest3Plugin;
impl Plugin for CampaignWettbewerbQuest3Plugin {
    fn build(&self, app: &mut App) {
        app.add_systems(
            OnEnter(CampaignWettbewerbState::InRound),
            (clear_hand, quest3_setup)
                .run_if(in_state(CampaignWettbewerb::Quest3))
                .after(generate_hand)
                .before(simulate_bot),
        );
        // add setup OnEnter(Quest3::Hole), but in Update schedule with run conditions to allow the usage of before and after.
        app.add_systems(
            Update,
            quest3_setup2
                .run_if(in_state(Quest3::Hole))
                .run_if(state_changed::<Quest3>)
                .after(GameSimSet::AfterSim)
                .before(GameSimSet::NextAuto),
        );
        app.add_systems(
            Update,
            clear_hand
                .run_if(on_event::<Redraw>)
                .run_if(in_state(CampaignWettbewerb::Quest3))
                .in_set(GameSimSet::AfterSim)
                .after(process_manual_redraw),
        );
    }
}

fn quest3_setup(mut commands: Commands, display_mode: Res<State<DisplayMode>>) {
    commands.spawn(TileLoaded).queue(
        AddTile::new(TileType::Forest, CubeCoordinate::default())
            .without_scene_cond(*display_mode.get() == DisplayMode::Headless)
            .with_data(None),
    );
}

fn clear_hand(mut hand: ResMut<Hand>) {
    hand.items = vec![TileType::Forest; 5];
}

fn quest3_setup2(
    mut commands: Commands,
    build_area: Query<&BuildArea>,
    display_mode: Res<State<DisplayMode>>,
) {
    let Ok(build_area) = build_area.get_single() else {
        return;
    };
    for coord in build_area.center.get_area(3) {
        commands.spawn(TileLoaded).queue(
            AddTile::new(TileType::Forest, coord)
                .without_scene_cond(*display_mode.get() == DisplayMode::Headless)
                .with_data(None),
        );
    }
    let mut iter = build_area.center.get_area(2);
    let coord = iter
        .nth(fastrand::usize(0..iter.size_hint().0))
        .expect("Should be in range. Size_hint is assumed to be correct.");
    commands.queue(RemoveTile::new(coord));
}

fn test_forest(
    map: Res<Map>,
    forest: Query<&CubeCoordinate, With<Enum!(TileType::Forest)>>,
) -> bool {
    for coord in forest.iter() {
        let forest_neighbors = map
            .get_neighbors(*coord)
            .filter(|e| forest.contains(*e))
            .count();
        if forest_neighbors >= 6 {
            return true;
        }
    }
    false
}

fn test_forest2(
    forest: Query<Entity, With<Enum!(TileType::Forest)>>,
    build_area: Query<&BuildArea>,
) -> bool {
    let Ok(build_area) = build_area.get_single() else {
        return true;
    };
    // area iterator ignores center tile
    forest.iter().count() > build_area.center.get_area(3).size_hint().0
}

define_level!(
    CampaignWettbewerb::Quest3,
    story: [
        {
            id: "intro"
        },
        {
            ref: Forest,
            id: "story-forest",
            goal: test_forest
        },
        {
            ref: Hole,
            id: "story-hole",
            goal: test_forest2
        },
        {
            id: "done"
        }
    ]
);
