use crate::{
    coordinates::CubeCoordinate,
    game::{
        bot::challenge_bot::simulate_bot,
        gamemodes::campaign::levels::CampaignWettbewerb,
        hand::{generate_hand, Hand},
        map::Map,
        placement_reclaim::TileLoaded,
        rewards::RewardEvent,
        tiles::{tile_type_filters, AddTile, RemoveTile, TileType},
    },
    DisplayMode,
};
use bevy::prelude::*;
use bevy_enum_filter::Enum;

use super::CampaignWettbewerbState;

pub struct CampaignWettbewerbQuest2Plugin;
impl Plugin for CampaignWettbewerbQuest2Plugin {
    fn build(&self, app: &mut App) {
        app.add_systems(
            OnEnter(CampaignWettbewerbState::InRound),
            (clear_hand, quest2_setup)
                .run_if(in_state(CampaignWettbewerb::Quest2))
                .after(generate_hand)
                .before(simulate_bot),
        );
    }
}

fn quest2_setup(
    mut commands: Commands,
    display_mode: Res<State<DisplayMode>>,
    placed: Query<&CubeCoordinate, With<TileType>>,
) {
    for coord in placed.iter() {
        commands.queue(RemoveTile::new(*coord));
    }
    let mut iter = CubeCoordinate::default().get_area(2);
    let id = fastrand::usize(1..iter.size_hint().0);
    let center = iter
        .nth(id)
        .expect("Should be in range. Size_hint is assumed to be correct.");

    let direction = fastrand::usize(0..6);
    let mut dir_iter = center.get_neighbors().cycle().skip(direction);
    let first = dir_iter.next().expect("cycle must never end");
    let second = dir_iter.nth(2).expect("cycle must never end");
    for coord in [first, second] {
        commands.spawn(TileLoaded).queue(
            AddTile::new(TileType::Wheat, coord)
                .without_scene_cond(*display_mode.get() == DisplayMode::Headless)
                .with_data(None),
        );
    }

    commands.spawn_empty().queue(
        AddTile::new(TileType::Wheat, CubeCoordinate::default() - center)
            .without_scene_cond(*display_mode.get() == DisplayMode::Headless)
            .with_data(None),
    );
}

fn clear_hand(mut hand: ResMut<Hand>) {
    hand.items = Vec::new();
    hand.selected = Some(0);
}

fn test_wheat(map: Res<Map>, wheat: Query<&CubeCoordinate, With<Enum!(TileType::Wheat)>>) -> bool {
    for coord in wheat.iter() {
        let wheat_neighbors = map
            .get_neighbors(*coord)
            .filter(|e| wheat.contains(*e))
            .count();
        if wheat_neighbors >= 2 {
            return true;
        }
    }
    false
}

define_level!(
    CampaignWettbewerb::Quest2,
    story: [
        {
            id: "intro"
        },
        {
            ref: Wheat,
            id: "story-wheat",
            goal: test_wheat
        },
        {
            ref: Rewards,
            id: "story-rewards",
            goal: on_event::<RewardEvent>
        },
        {
            id: "done"
        }
    ]
);
