use crate::{
    game::{
        gamemodes::{campaign::CampaignLevelProgress, GameModeSubStates},
        round::RoundSystemSupport,
    },
    DisplayMode,
};
use bevy::prelude::*;
use quest2::CampaignWettbewerbQuest2Plugin;
use quest3::CampaignWettbewerbQuest3Plugin;
use quest4::CampaignWettbewerbQuest4Plugin;
use quest5::CampaignWettbewerbQuest5Plugin;
use weihnachten::CampaignWettbewerbWeihnachtenPlugin;

use super::CampaignGroups;

pub mod quest2;
pub mod quest3;
pub mod quest4;
pub mod quest5;
pub mod weihnachten;

pub struct CampaignWettbewerbPlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for CampaignWettbewerbPlugin {
    fn build(&self, app: &mut App) {
        // register basic challenge system
        app.add_computed_state::<CampaignWettbewerbStage>();
        app.add_sub_state::<CampaignWettbewerbState>();

        // add more levels to the group here:
        app.add_plugins(CampaignWettbewerbWeihnachtenPlugin);
        app.add_plugins(CampaignWettbewerbQuest2Plugin);
        app.add_plugins(CampaignWettbewerbQuest3Plugin);
        app.add_plugins(CampaignWettbewerbQuest4Plugin);
        app.add_plugins(CampaignWettbewerbQuest5Plugin);
    }
}

/// Meta state used to combine the group and level progress
/// should only exist when the game is in this level group
#[derive(Debug, Clone, Copy, PartialEq, Eq, Hash)]
pub enum CampaignWettbewerbStage {
    /// the level is in the Intro or Outro state
    Meta,
    /// the level is in the Running state
    InGame,
}

impl ComputedStates for CampaignWettbewerbStage {
    type SourceStates = (CampaignGroups, CampaignLevelProgress);

    fn compute(sources: Self::SourceStates) -> Option<Self> {
        if sources.0 == CampaignGroups::CampaignWettbewerb {
            return Some(match sources.1 {
                CampaignLevelProgress::Intro | CampaignLevelProgress::Outro => Self::Meta,
                CampaignLevelProgress::Running => Self::InGame,
            });
        }
        None
    }
}

// base state system
#[derive(Debug, Clone, Copy, Default, PartialEq, Eq, Hash, SubStates)]
#[source(CampaignWettbewerbStage = CampaignWettbewerbStage::InGame)]
pub enum CampaignWettbewerbState {
    /// the player is currently in a round
    /// and can place tiles and redraw cards
    #[default]
    InRound,
    /// the player is between two rounds
    BetweenRounds,
    /// the player lost the game
    GameOver,
    /// Immutable game view
    /// the player can still use the camera to view the world
    Spectate,
}

impl GameModeSubStates for CampaignWettbewerbState {}

impl RoundSystemSupport for CampaignWettbewerbState {
    fn get_gameover_state() -> Self {
        Self::GameOver
    }
    fn get_phaseend_state() -> Self {
        Self::BetweenRounds
    }
    fn get_inround_state() -> Self {
        Self::InRound
    }
    fn get_spectate_state() -> Option<Self> {
        Some(Self::Spectate)
    }
}
