use crate::{
    coordinates::CubeCoordinate,
    game::{
        gamemodes::campaign::{levels::CampaignTutorial, CampaignLevelProgress},
        hand::Hand,
        map::Map,
        placement_reclaim::TileLoaded,
        tiles::{tile_type_filters, AddTile, TileType},
        time::pause_on_focus_loss,
    },
    DisplayMode,
};
use bevy::{
    app::{App, Plugin},
    prelude::SubStates,
};
use bevy_enum_filter::Enum;

use bevy::prelude::*;

pub struct CampaignEarlyGameTutorialPlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for CampaignEarlyGameTutorialPlugin {
    fn build(&self, app: &mut App) {
        app.add_sub_state::<EarlyGameTutorialState>();

        if self.display_mode == DisplayMode::Graphic {
            app.add_systems(OnEnter(EarlyGameTutorialState::InGame), setup_world);
            app.add_systems(OnEnter(EarlyGameTutorialState::InGame), setup_hand_forest);
            app.add_systems(OnEnter(EarlyGameTutorial::Wheat), setup_hand_wheat);
            app.add_systems(OnEnter(EarlyGameTutorial::Houses), setup_hand_houses);

            app.add_systems(
                Update,
                pause_on_focus_loss
                    .run_if(in_state(EarlyGameTutorialState::InGame))
                    .run_if(in_state(CampaignLevelProgress::Running)),
            );
        }
    }
}

#[derive(SubStates, Debug, Hash, PartialEq, Clone, Eq, Default)]
#[source(CampaignTutorial = CampaignTutorial::EarlyGameTutorial)]
pub enum EarlyGameTutorialState {
    #[default]
    InGame,
}

// check if one of the wheat clusters has optimum production (or optimum size)
pub fn wheat_placed(
    optimum_size: usize,
) -> impl FnMut(Res<Map>, Query<&CubeCoordinate, With<Enum!(TileType::Wheat)>>) -> bool + Clone {
    move |le_map, wheat_tiles| {
        let x = le_map.depth_search_collect(CubeCoordinate::new(1, -1, 0), |_, e| {
            wheat_tiles.contains(e)
        });
        x.len() == optimum_size
    }
}

// check if certain forest tile has optimum production (only one possibility)
pub fn forest_placed(
) -> impl FnMut(Res<Map>, Query<&CubeCoordinate, With<Enum!(TileType::Forest)>>) -> bool + Clone {
    move |le_map, forest_tiles| {
        let required_forest_tiles = [
            CubeCoordinate::new(2, 0, -2), // center tile
            CubeCoordinate::new(1, 0, -1),
            CubeCoordinate::new(2, -1, -1),
            CubeCoordinate::new(3, -1, -2),
            CubeCoordinate::new(3, 0, -3),
            CubeCoordinate::new(2, 1, -3),
            CubeCoordinate::new(1, 1, -2),
        ];

        for coord in required_forest_tiles {
            if let Some(x) = le_map.get(coord) {
                if !forest_tiles.contains(x) {
                    return false;
                }
            } else {
                return false;
            }
        }
        true
    }
}

// requires houses to be placed at two coordinates
pub fn houses_placed(
) -> impl FnMut(Res<Map>, Query<&CubeCoordinate, With<Enum!(TileType::SmallHouse)>>) -> bool + Clone
{
    move |le_map, house_tiles| {
        let required_house_tiles = [CubeCoordinate::new(1, 2, -3), CubeCoordinate::new(0, 3, -3)];

        for coord in required_house_tiles {
            if let Some(x) = le_map.get(coord) {
                if !house_tiles.contains(x) {
                    return false;
                }
            } else {
                return false;
            }
        }
        true
    }
}

fn setup_world(mut commands: Commands, display_mode: Res<State<DisplayMode>>) {
    let prebuild_world = [
        (TileType::Beehive, CubeCoordinate::new(0, 0, 0)),
        (TileType::Forest, CubeCoordinate::new(1, 0, -1)),
        (TileType::Forest, CubeCoordinate::new(2, -1, -1)),
        (TileType::Wheat, CubeCoordinate::new(1, -1, 0)),
        (TileType::Wheat, CubeCoordinate::new(0, -1, 1)),
        (TileType::Wheat, CubeCoordinate::new(1, -2, 1)),
        (TileType::Wheat, CubeCoordinate::new(2, -2, 0)),
        (TileType::Wheat, CubeCoordinate::new(3, -2, -1)),
        (TileType::Wheat, CubeCoordinate::new(3, -2, -1)),
        (TileType::Wheat, CubeCoordinate::new(-1, 1, 0)),
        (TileType::Wheat, CubeCoordinate::new(-1, 2, -1)),
        (TileType::Wheat, CubeCoordinate::new(-2, 2, 0)),
        (TileType::Wheat, CubeCoordinate::new(-2, 3, -1)),
        (TileType::Windmill, CubeCoordinate::new(2, -3, 1)),
        (TileType::Windmill, CubeCoordinate::new(-1, 3, -2)),
        (TileType::SmallHouse, CubeCoordinate::new(0, 1, -1)),
        (TileType::DoubleHouse, CubeCoordinate::new(0, 2, -2)),
    ];

    for (tile_type, coords) in prebuild_world {
        commands
            .spawn_empty()
            .queue(
                AddTile::new(tile_type, coords)
                    .without_scene_cond(*display_mode.get() == DisplayMode::Headless)
                    .with_data(None),
            )
            .insert(TileLoaded);
    }
}

fn setup_hand_wheat(mut hand: ResMut<Hand>) {
    hand.items = vec![
        TileType::Wheat,
        TileType::Wheat,
        TileType::Grass,
        TileType::Wheat,
        TileType::Wheat,
    ];
    hand.selected = Some(0);
}

fn setup_hand_forest(mut hand: ResMut<Hand>) {
    hand.items = vec![
        TileType::Forest,
        TileType::Forest,
        TileType::Forest,
        TileType::Forest,
        TileType::Forest,
    ];
    hand.selected = Some(0);
}

fn setup_hand_houses(mut hand: ResMut<Hand>) {
    hand.reset();
    hand.items = vec![
        TileType::SmallHouse,
        TileType::Grass,
        TileType::StoneRocks,
        TileType::SmallHouse,
        TileType::Forest,
    ];
    hand.selected = Some(0);
}

define_level!(
CampaignTutorial::EarlyGameTutorial,
story: [
    {
        id: "early_game_intro"
    },
    {
        ref: Forest,
        id: "early_game_forest",
        goal: forest_placed()
    },
    {
        ref: Wheat,
        id: "early_game_wheat",
        goal: wheat_placed(9)
    },
    {
        ref: Houses,
        id: "early_game_houses",
        goal: houses_placed()
    },
    {
        id: "early_game_done"
    }
]);
