use crate::coordinates::CubeCoordinate;
use crate::game::controls::PanOrbitCam;
use crate::game::gamemodes::campaign::levels::CampaignTutorial;
use crate::game::gamemodes::campaign::CampaignLevelProgress;
use crate::game::tiles::{AddTile, TileType};
use crate::game::time::pause_on_focus_loss;
use crate::DisplayMode;
use bevy::prelude::*;

pub struct CampaignCameraTutorialPlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for CampaignCameraTutorialPlugin {
    fn build(&self, app: &mut App) {
        app.add_sub_state::<CameraTutorialStates>();
        if self.display_mode == DisplayMode::Graphic {
            app.add_systems(OnEnter(CameraTutorial::Move), reset_cam_state);
            app.add_systems(OnEnter(CameraTutorial::Rotate), reset_cam_state);
            app.add_systems(OnEnter(CameraTutorial::Zoom), reset_cam_state);

            app.add_systems(OnEnter(CameraTutorialStates::InGame), setup_world);

            app.add_systems(
                Update,
                pause_on_focus_loss
                    .run_if(in_state(CameraTutorialStates::InGame))
                    .run_if(in_state(CampaignLevelProgress::Running)),
            );
        }
    }
}

fn setup_world(mut commands: Commands, display_mode: Res<State<DisplayMode>>) {
    commands.spawn_empty().queue(
        AddTile::new(TileType::Wheat, CubeCoordinate::new(0, 0, 0))
            .without_scene_cond(*display_mode.get() == DisplayMode::Headless)
            .with_data(None),
    );
}

#[derive(SubStates, Debug, Hash, PartialEq, Clone, Eq, Default)]
#[source(CampaignTutorial = CampaignTutorial::CameraTutorial)]
pub enum CameraTutorialStates {
    #[default]
    InGame,
}

/// resets the camera state
fn reset_cam_state(mut cam_query: Query<&mut PanOrbitCam>) {
    if !cam_query.is_empty() {
        let mut cam = cam_query.single_mut();
        *cam = PanOrbitCam::default();
    }
}

/// Generates a [`Condition`](bevy_ecs::prelude::Condition)-satisfying closure that returns `true`
/// if the camera was moved the given distance since `store_cam_state` was called last
pub fn cam_move_delta(target_dist: f32) -> impl FnMut(Query<&PanOrbitCam>) -> bool + Clone {
    move |cam_query| {
        let cam = cam_query.get_single().copied().unwrap_or_default();
        let dist = cam.focus.distance(PanOrbitCam::default().focus).abs();
        dist >= target_dist
    }
}
/// Generates a [`Condition`](bevy_ecs::prelude::Condition)-satisfying closure that returns `true`
/// if the camera was rotated the given distance since `store_cam_state` was called last
pub fn cam_rotate_delta(target_dist: f32) -> impl FnMut(Query<&PanOrbitCam>) -> bool + Clone {
    move |cam_query| {
        let cam = cam_query.get_single().copied().unwrap_or_default();
        let dist = cam.orbit.distance(PanOrbitCam::default().orbit).abs();
        dist >= target_dist
    }
}
/// Generates a [`Condition`](bevy_ecs::prelude::Condition)-satisfying closure that returns `true`
/// if the camera zoom was changed by at least `target_dist` since `store_cam_state` was called last
pub fn cam_zoom_delta(target_dist: f32) -> impl FnMut(Query<&PanOrbitCam>) -> bool + Clone {
    move |cam_query| {
        let cam = cam_query.get_single().copied().unwrap_or_default();
        let dist = (cam.radius - PanOrbitCam::default().radius).abs();
        dist >= target_dist
    }
}

define_level!(
CampaignTutorial::CameraTutorial,
story: [
    {
        id: "cam_intro"
    },
    {
        ref: Move,
        id: "cam_move",
        goal: cam_move_delta(2.0)
    },
    {
        ref: Rotate,
        id: "cam_rotate",
        goal: cam_rotate_delta(0.8)
    },
    {
        ref: Zoom,
        id: "cam_zoom",
        goal: cam_zoom_delta(10.0)
    },
    {
        id: "cam_done"
    }
]);
