use bevy::{
    app::{App, Plugin},
    prelude::SubStates,
};

use crate::{
    game::{
        gamemodes::campaign::{levels::CampaignTutorial, CampaignLevelProgress},
        placement_reclaim::{PlacementTimer, TileLoaded},
        resources::{TilePlacedEvent, TileReclaimedEvent},
        rewards::RewardEvent,
        time::pause_on_focus_loss,
    },
    DisplayMode,
};

use bevy::prelude::*;

pub struct CampaignTilesTutorialPlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for CampaignTilesTutorialPlugin {
    fn build(&self, app: &mut App) {
        app.add_sub_state::<TilesTutorialState>();
        app.add_systems(
            Update,
            reset_tile_animation.run_if(in_state(TilesTutorial::Reward)),
        );

        if let DisplayMode::Graphic = self.display_mode {
            app.add_systems(
                Update,
                pause_on_focus_loss
                    .run_if(in_state(TilesTutorialState::InGame))
                    .run_if(in_state(CampaignLevelProgress::Running)),
            );
        }
    }
}

#[derive(SubStates, Debug, Hash, PartialEq, Clone, Eq, Default)]
#[source(CampaignTutorial = CampaignTutorial::TilesTutorial)]
pub enum TilesTutorialState {
    #[default]
    InGame,
}

pub fn reset_tile_animation(mut commands: Commands, tiles: Query<Entity, With<PlacementTimer>>) {
    for entity in &tiles {
        commands
            .entity(entity)
            .remove::<PlacementTimer>()
            .insert(TileLoaded);
    }
}

define_level!(
CampaignTutorial::TilesTutorial,
story: [
    {
        id: "tiles_intro"
    },
    {
        ref: Place,
        id: "tiles_place",
        goal: on_event::<TilePlacedEvent>
    },
    {
        ref: Reclaim,
        id: "tiles_reclaim",
        goal: on_event::<TileReclaimedEvent>
    },
    {
        ref: Reward,
        id: "tiles_reward",
        goal: on_event::<RewardEvent>
    },
    {
        id: "tiles_done"
    }
]);
