use bevy::{ecs::system::RunSystemOnce, prelude::*};
use challenge::{easy::ChallengeEasy, CampaignChallengePlugin};
#[cfg(feature = "graphics")]
use tutorial::basic_tile_functions::TilesTutorial;
#[cfg(feature = "graphics")]
use tutorial::camera::CameraTutorial;
#[cfg(feature = "graphics")]
use tutorial::early_game_tiles::EarlyGameTutorial;
use tutorial::CampaignTutorialPlugin;
use wettbewerb::CampaignWettbewerbPlugin;

use crate::{
    game::{
        bot::{challenge_bot::start_challenge_bot_game, creative_bot::start_creative_bot_game},
        gamemodes::campaign::levels::wettbewerb::{
            quest2::Quest2, quest3::Quest3, quest4::Quest4, quest5::Quest5,
            weihnachten::WettbewerbWeihnachten,
        },
    },
    DisplayMode,
};

pub mod challenge;
pub mod tutorial;
pub mod wettbewerb;

pub struct CampaignLevelsPlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for CampaignLevelsPlugin {
    fn build(&self, app: &mut App) {
        app.add_plugins(CampaignTutorialPlugin {
            display_mode: self.display_mode,
        });
        app.add_plugins(CampaignChallengePlugin {
            display_mode: self.display_mode,
        });
        app.add_plugins(CampaignWettbewerbPlugin {
            display_mode: self.display_mode,
        });
    }
}

// Define all known groups and levels
#[cfg(feature = "graphics")]
campaign_groups!({
   "tutorial": {
       ref: CampaignTutorial,
       levels: {
           "camera": CameraTutorial,
           "basic_tile_functions": TilesTutorial,
           "early_game_tiles": EarlyGameTutorial
       }
   },
    "challenge": {
        ref: CampaignChallenge,
        levels: {
            "easy": ChallengeEasy
        }
    },
    "wettbewerb": {
        ref: CampaignWettbewerb,
        levels: {
            "quest2": Quest2,
            "quest3": Quest3,
            "quest4": Quest4,
            "quest5": Quest5,
            "weihnachten": WettbewerbWeihnachten,
        }
    }
});

#[cfg(not(feature = "graphics"))]
// NOTE: the tutorial group has to be readded, once it contains
// a level that is supported in mini mode
campaign_groups!({
    "challenge": {
        ref: CampaignChallenge,
        levels: {
            "easy": ChallengeEasy
        }
    },
    "wettbewerb": {
        ref: CampaignWettbewerb,
        levels: {
            "weihnachten": WettbewerbWeihnachten,
            "quest2": Quest2,
            "quest3": Quest3,
            "quest4": Quest4,
            "quest5": Quest5,
        }
    }
});

/// type of bot support for a campaign group/level
#[derive(Debug)]
pub enum CampaignBotType {
    /// creative like campaign level/bot
    Creative,
    /// challenge like campaign level/bot
    Challenge,
}

impl CampaignGroupWrapper {
    /// returns the type of bot support (if any) for a given level
    /// matching may be done on group or level basis
    pub fn get_bot_type(&self) -> Option<CampaignBotType> {
        #[allow(unreachable_patterns)]
        match self {
            // add campaign levels that support bots here,
            // with the type of bot that is supported
            // NOTE: You also gave to register the states in
            // game::bot::{challenge/creative}_bot.rs and
            // game::controls::bot_placer.rs
            Self::CampaignChallenge(_) => Some(CampaignBotType::Challenge),
            Self::CampaignWettbewerb(_) => Some(CampaignBotType::Challenge),
            _ => None,
        }
    }
    /// try to initialize the bot
    /// returns false if bot support isn't available for the given level
    pub fn init_bot(&self, w: &mut World) -> bool {
        if let Some(kind) = self.get_bot_type() {
            match kind {
                CampaignBotType::Challenge => {
                    w.run_system_once(start_challenge_bot_game).unwrap();
                }
                CampaignBotType::Creative => {
                    w.run_system_once(start_creative_bot_game).unwrap();
                }
            }
            return true;
        }
        false
    }
}
