use crate::{
    game::{
        gamemodes::{campaign::CampaignLevelProgress, GameModeSubStates},
        round::RoundSystemSupport,
    },
    DisplayMode,
};
use bevy::prelude::*;
use easy::CampaignChallengeEasyPlugin;

use super::CampaignGroups;

pub mod easy;

pub struct CampaignChallengePlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for CampaignChallengePlugin {
    fn build(&self, app: &mut App) {
        // register basic challenge system
        app.add_computed_state::<CampaignChallengeStage>();
        app.add_sub_state::<CampaignChallengeState>();

        // add more levels to the group here:
        app.add_plugins(CampaignChallengeEasyPlugin {
            display_mode: self.display_mode,
        });
    }
}

/// Meta state used to combine the group and level progress
/// should only exist when the game is in this level group
#[derive(Debug, Clone, Copy, PartialEq, Eq, Hash)]
pub enum CampaignChallengeStage {
    /// the level is in the Intro or Outro state
    Meta,
    /// the level is in the Running state
    InGame,
}

impl ComputedStates for CampaignChallengeStage {
    type SourceStates = (CampaignGroups, CampaignLevelProgress);

    fn compute(sources: Self::SourceStates) -> Option<Self> {
        if sources.0 == CampaignGroups::CampaignChallenge {
            return Some(match sources.1 {
                CampaignLevelProgress::Intro | CampaignLevelProgress::Outro => Self::Meta,
                CampaignLevelProgress::Running => Self::InGame,
            });
        }
        None
    }
}

// base state system
#[derive(Debug, Clone, Copy, Default, PartialEq, Eq, Hash, SubStates)]
#[source(CampaignChallengeStage = CampaignChallengeStage::InGame)]
pub enum CampaignChallengeState {
    /// the player is currently in a round
    /// and can place tiles and redraw cards
    #[default]
    InRound,
    /// the player is between two rounds
    BetweenRounds,
    /// the player lost the game
    GameOver,
    /// Immutable game view
    /// the player can still use the camera to view the world
    Spectate,
}

impl GameModeSubStates for CampaignChallengeState {}

impl RoundSystemSupport for CampaignChallengeState {
    fn get_gameover_state() -> Self {
        Self::GameOver
    }
    fn get_phaseend_state() -> Self {
        Self::BetweenRounds
    }
    fn get_inround_state() -> Self {
        Self::InRound
    }
    fn get_spectate_state() -> Option<Self> {
        Some(Self::Spectate)
    }
}
