use bevy::{ecs::system::SystemParam, prelude::*, window::PrimaryWindow};
use bevy_egui::EguiContexts;

use crate::{
    coordinates::{distance_ray_point, CubeCoordinate},
    game::{
        build_area::BUILD_PLATE_HEIGHT,
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState,
                tutorial::{
                    basic_tile_functions::TilesTutorialState,
                    early_game_tiles::EarlyGameTutorialState,
                },
                wettbewerb::CampaignWettbewerbState,
            },
            challenge::ChallengeGameState,
            zen::ZenGameState,
        },
        hand::RandomCore,
        map::TileSaveData,
        placement_reclaim::{handle_tile_reclaim, PlacementTimer},
        resources::{TilePlacedEvent, TileReclaimedEvent},
        rewards::RewardReqEvent,
        tiles::{AddTile, TileType},
        GamePauseState,
    },
    settings::ActiveSettingsBank,
    ui::{is_using_egui, UiClickEvent, WasUsingEgui},
};

use super::{
    cam::{handle_mouse_input, InputInfo},
    keyboard_placer::Cursor,
    CreativeInventory, GameData, PanOrbitCam, PlacerSet,
};

/// Radius of clickable sphere around reward center.
const REWARDS_MAX_CLICK_OFFSET: f32 = 0.2;

macro_rules! define_placer_plugin {
    ($({
        // state in which placer should be generated
        state: $state:expr,
    }),*) => {
        pub struct PlacerPlugin;
        impl Plugin for PlacerPlugin {
            fn build(&self, app: &mut App) {
                app.add_event::<TileClickEvent>();
                use crate::prelude::never;

                app.add_systems(
                    Update,
                    placer
                        .in_set(PlacerSet::Placer)
                        .before(handle_mouse_input)
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(in_state(GamePauseState::Running)),
                );

            }
        }
    }
}

define_placer_plugin![
    {
        state: ChallengeGameState::InRound,
    },
    {
        state: CampaignChallengeState::InRound,
    },
    {
        state: CampaignWettbewerbState::InRound,
    },
    {
        state: ZenGameState::InRound,
    },
    {
        state: TilesTutorialState::InGame,
    },
    {
        state: EarlyGameTutorialState::InGame,
    }
];

#[derive(Event, Debug, Clone, Copy)]
/// Will be called if tile was clicked on the ui. (This is visual only)
pub struct TileClickEvent {
    /// The CubeCoordinates of the clicked Tile
    pub tile_pos: CubeCoordinate,
}

#[derive(SystemParam)]
pub struct InputData<'w> {
    input_info: Res<'w, InputInfo>,
    // data from bevy
    input_mouse: Res<'w, ButtonInput<MouseButton>>,
    touches: Res<'w, Touches>,
}

#[derive(SystemParam)]
pub struct PlacerEvents<'w> {
    pub ev_calc_place: EventWriter<'w, TilePlacedEvent>,
    pub ev_calc_reclaim: EventWriter<'w, TileReclaimedEvent>,
    pub tile_click_event: EventWriter<'w, TileClickEvent>,
    pub reward_event: EventWriter<'w, RewardReqEvent>,
}

#[allow(clippy::too_many_arguments)]
pub fn placer(
    mut contexts: EguiContexts,
    was_using: Res<WasUsingEgui>,
    mut commands: Commands,
    input_data: InputData,
    mut events: PlacerEvents,
    mut game: GameData,
    settings: Res<ActiveSettingsBank>,
    cam_query: Query<(&Camera, &GlobalTransform), With<PanOrbitCam>>,
    q_window: Query<&Window, With<PrimaryWindow>>,
    loading_tiles: Query<(&TileType, &CubeCoordinate), With<PlacementTimer>>,
    mut rng: ResMut<RandomCore>,
    mut cursor: Query<&mut Visibility, With<Cursor>>,
    debug_inv: Option<Res<CreativeInventory>>,
    mut click_event: EventWriter<UiClickEvent>,
) {
    let click_button = settings.mouse.place;

    if is_using_egui(&mut contexts) || was_using.using {
        return;
    }

    let Ok(build_area) = game.build_area.get_single() else {
        return;
    };

    let window = q_window.single();
    let (camera, camera_transform) = cam_query.single();

    // get click position
    let cursor_position = if input_data.input_mouse.just_released(click_button) {
        if input_data.input_info.moved.length() > 2.0 {
            return;
        }
        let Some(pos) = window.cursor_position() else {
            return;
        };
        pos
    } else {
        if !input_data.input_info.touch_placeable {
            return;
        }
        if input_data.touches.iter().count() != 0
            || input_data.touches.iter_just_released().count() != 1
        {
            return;
        }
        let touch = input_data.touches.iter_just_released().next().unwrap();
        if input_data.touches.just_released(touch.id()) && touch.distance().length() > 2.0 {
            return;
        }
        touch.position()
    };

    let Ok(ray) = camera.viewport_to_world(camera_transform, cursor_position) else {
        return;
    };

    if let Ok(mut cursor) = cursor.get_single_mut() {
        *cursor = Visibility::Hidden;
    }

    // try click rewards
    let mut interacted_with_rewards = false;
    for (transform, coord) in game.rewards.iter() {
        let pos = transform.translation;
        let distance = distance_ray_point(ray, pos);
        if distance > REWARDS_MAX_CLICK_OFFSET {
            continue;
        }
        // event will take care of deleting the reward
        events.reward_event.send(RewardReqEvent(*coord));
        interacted_with_rewards = true;
        click_event.send(UiClickEvent);
    }

    if interacted_with_rewards {
        return;
    }

    // try place tile
    let plane = InfinitePlane3d::new(Vec3::Y);
    let Some(distance) = ray.intersect_plane(Vec3::new(0.0, BUILD_PLATE_HEIGHT, 0.0), plane) else {
        return;
    };
    let global_cursor: CubeCoordinate = ray.get_point(distance).into();

    let mut tile = None;
    let mut hand_index = None;

    if let Some(Some(st)) = debug_inv.and_then(|inv| inv.slots.get(inv.selected).cloned()) {
        tile = Some(st);
    } else if let Some(selected) = game.hand.selected {
        if let Some(st) = game.hand.items.get(selected) {
            tile = Some(*st);
            hand_index = game.hand.selected;
        }
    };
    let Some(tile) = tile else {
        // reclaim tile or send click event, if tile was hit
        if let Some(e) = game.map.get(global_cursor) {
            if handle_tile_reclaim(
                &mut commands,
                e,
                &loading_tiles,
                &mut game.hand,
                &mut game.metrics,
                &game.rate,
            ) {
                events
                    .ev_calc_reclaim
                    .send(TileReclaimedEvent(global_cursor));
                return;
            }
            events.tile_click_event.send(TileClickEvent {
                tile_pos: global_cursor,
            });
        }
        return;
    };

    // placer rules only apply to cards from the hand
    // debug placer tiles can always be placed anywhere
    if hand_index.is_some() {
        // check if global_cursor is inside the range of the build area
        if !build_area.check(global_cursor) {
            return;
        }

        // check if no tile is at that position already
        if let Some(e) = game.map.get(global_cursor) {
            if handle_tile_reclaim(
                &mut commands,
                e,
                &loading_tiles,
                &mut game.hand,
                &mut game.metrics,
                &game.rate,
            ) {
                events
                    .ev_calc_reclaim
                    .send(TileReclaimedEvent(global_cursor));
                return;
            }
            events.tile_click_event.send(TileClickEvent {
                tile_pos: global_cursor,
            });
            return;
        }

        // check if tile is adjacent to another tile (when at least one exists)
        if !game.map.is_empty() && game.map.get_neighbors(global_cursor).count() == 0 {
            return;
        }
    }

    commands
        .spawn_empty()
        .queue(AddTile::new(tile, global_cursor));

    {
        const RAND_GARBAGE_LENGTH: usize = 73;
        let mut random_garbage = [0u8; RAND_GARBAGE_LENGTH];
        let CubeCoordinate { q, r, s } = global_cursor;
        for coord in [q, r, s] {
            let count = (coord + rng.u32(..) as isize) as usize % RAND_GARBAGE_LENGTH;
            let window = &mut random_garbage[0..count];
            rng.fill(window);
        }
    }

    // tile was selected from hand
    // so we have to remove the card
    // as it has been used
    if let Some(index) = hand_index {
        game.hand.items.remove(index);
        if let Some(si) = game.hand.selected {
            // select last available slot,
            // if there is no card after the one we just placed
            if si >= game.hand.items.len() {
                game.hand.select_previous_slot();
            }
        }
    }

    // signal consumption
    events
        .ev_calc_place
        .send(TilePlacedEvent(TileSaveData::new(global_cursor, tile)));
}
