use bevy::{ecs::system::SystemParam, prelude::*, window::PrimaryWindow};
use bevy_egui::EguiContexts;

use crate::{
    coordinates::CubeCoordinate,
    game::{
        build_area::BUILD_PLATE_HEIGHT,
        controls::keyboard_placer::Cursor,
        gamemodes::{
            creative::{CreativeGameState, CREATIVE_INV_SLOT_COUNT},
            GameMode,
        },
        map::{Map, TileSaveData},
        resources::{TilePlacedEvent, TileReclaimedEvent},
        tiles::{AddTile, RemoveTile},
        GamePauseState,
    },
    settings::ActiveSettingsBank,
    ui::is_using_egui,
};

use super::{
    cam::{handle_mouse_input, handle_touch_input, InputInfo},
    placer::TileClickEvent,
    CreativeInventory, PanOrbitCam, PlacerSet,
};

macro_rules! define_creative_placer_plugin {
    ($({
        // in which state to run the creative_placer
        state: $state:expr,
        // number of slots in inventory
        inventory_size: $inv_size:expr,
    }),*) => {
        pub struct CreativePlacerPlugin;
        impl Plugin for CreativePlacerPlugin {
            fn build(&self, app: &mut App) {

                app.add_event::<TileClickEvent>();
                $(
                    app.add_systems(
                        Update,
                        creative_placer
                            .in_set(PlacerSet::Placer)
                            .before(handle_mouse_input)
                            .before(handle_touch_input)
                            .run_if(in_state($state))
                            .run_if(in_state(GamePauseState::Running)),
                    );
                )*
            }
        }
    }
}

#[cfg(feature = "graphics")]
define_creative_placer_plugin![
    {
        state: CreativeGameState::InGame,
        inventory_size: CREATIVE_INV_SLOT_COUNT,
    }
];

#[derive(SystemParam)]
pub struct CreativePlacerEvents<'w> {
    pub calc_place: EventWriter<'w, TilePlacedEvent>,
    pub calc_reclaim: EventWriter<'w, TileReclaimedEvent>,
    pub tile_click: EventWriter<'w, TileClickEvent>,
}

macro_rules! define_creative_inventory_plugin {
    ($({
        // gamemode in which the inventory should be available
        gamemode: $gm:expr,
        // number of slots in inventory
        inventory_size: $inv_size:expr$(,)?
    }),*) => {
        pub struct CreativeInventoryPlugin;
        impl Plugin for CreativeInventoryPlugin {
            fn build(&self, app: &mut App) {
                $(
                    app.init_resource::<CreativeInventory>();

                    app.add_systems(
                        OnEnter($gm),
                        |mut inv: ResMut<CreativeInventory>| {
                            inv.selected = 0;
                            *inv.slots = vec![None; $inv_size];
                        }
                    );
                    // reset the inventory on exit
                    // to prevent the debug placer from being used in gamemodes
                    // that don't use it
                    app.add_systems(
                        OnExit($gm),
                        |mut inv: ResMut<CreativeInventory>| {
                            inv.selected = 0;
                            *inv.slots = vec![None; $inv_size];
                        }
                    );
                )*
            }
        }
    }
}

#[cfg(not(debug_assertions))]
define_creative_inventory_plugin![
    {
        gamemode: GameMode::Creative,
        inventory_size: CREATIVE_INV_SLOT_COUNT
    }
];
#[cfg(debug_assertions)]
define_creative_inventory_plugin![
    {
        gamemode: GameMode::Creative,
        inventory_size: CREATIVE_INV_SLOT_COUNT
    },
    {
        gamemode: GameMode::Challenge,
        inventory_size: crate::game::gamemodes::challenge::DEBUG_PLACER_SIZE
    }
];

#[derive(SystemParam)]
pub struct InputData<'w> {
    input_info: Res<'w, InputInfo>,
    // data from bevy
    input_mouse: Res<'w, ButtonInput<MouseButton>>,
    touches: Res<'w, Touches>,
}

/// how long to press down to destroy a tile
/// in f64 seconds
const TOUCH_LONG_PRESS_DURATION: f64 = 1.0;

#[allow(clippy::too_many_arguments)]
#[cfg(feature = "graphics")]
pub fn creative_placer(
    mut contexts: EguiContexts,
    was_using: Res<crate::ui::WasUsingEgui>,
    mut commands: Commands,
    input_data: InputData,
    settings: Res<ActiveSettingsBank>,
    cam_query: Query<(&Camera, &GlobalTransform), With<PanOrbitCam>>,
    q_window: Query<&Window, With<PrimaryWindow>>,
    inv: Res<CreativeInventory>,
    map: Res<Map>,
    time: Res<Time>,
    mut cursor: Query<&mut Visibility, With<Cursor>>,
    mut events: CreativePlacerEvents,
    cr_settings: Option<Res<crate::game::bot::creative_bot::CreativeSettings>>,
) {
    use crate::game::bot::creative_bot::creative_settings_cond;

    let place_button = settings.mouse.place;
    let destroy_button = settings.mouse.destroy;

    if is_using_egui(&mut contexts) || was_using.using {
        return;
    }

    let window = q_window.single();
    let (camera, camera_transform) = cam_query.single();

    let mut task_destroy = false;
    // get click position
    let mouse_place = input_data.input_mouse.just_released(place_button);
    let mouse_destroy = input_data.input_mouse.just_released(destroy_button);
    task_destroy |= mouse_destroy;
    let cursor_position = if mouse_destroy || mouse_place {
        if input_data.input_info.moved.length() > 2.0 {
            return;
        }
        let Some(pos) = window.cursor_position() else {
            return;
        };
        pos
    } else {
        if !input_data.input_info.touch_placeable {
            return;
        }

        let press_duration = time.elapsed_secs_f64() - input_data.input_info.touch_down;
        if press_duration > TOUCH_LONG_PRESS_DURATION {
            if input_data.touches.iter().count() != 1 {
                // long press only works with one touch
                // otherwise it will be considered a camera move action
                return;
            }
            let touch = input_data.touches.iter().next().unwrap();

            task_destroy = true;
            touch.position()
        } else {
            if input_data.touches.iter_just_released().count() != 1 {
                return;
            }
            let touch = input_data.touches.iter_just_released().next().unwrap();
            if touch.distance().length() > 2.0 {
                return;
            }

            touch.position()
        }
    };

    let Ok(ray) = camera.viewport_to_world(camera_transform, cursor_position) else {
        return;
    };

    if let Ok(mut cursor) = cursor.get_single_mut() {
        *cursor = Visibility::Hidden;
    }

    let plane = InfinitePlane3d::new(Vec3::Y);
    let Some(distance) = ray.intersect_plane(Vec3::new(0.0, BUILD_PLATE_HEIGHT, 0.0), plane) else {
        return;
    };
    let global_cursor: CubeCoordinate = ray.get_point(distance).into();

    if map.get(global_cursor).is_some() {
        if task_destroy {
            // try to remove tile
            if !creative_settings_cond::remove_tiles_ref(&cr_settings) {
                return;
            }
            commands.queue(RemoveTile::new(global_cursor));
            events.calc_reclaim.send(TileReclaimedEvent(global_cursor));
        } else {
            // show info dialog
            events.tile_click.send(TileClickEvent {
                tile_pos: global_cursor,
            });
        }
    } else if !task_destroy {
        // place new tile
        let Some(Some(tile)) = inv.slots.get(inv.selected) else {
            // no slot selected or slot empty
            return;
        };
        if !creative_settings_cond::place_tiles_ref(&cr_settings) {
            return;
        }
        commands
            .spawn_empty()
            .queue(AddTile::new(*tile, global_cursor));
        events
            .calc_place
            .send(TilePlacedEvent(TileSaveData::new(global_cursor, *tile)));
    }
}
