use crate::coordinates::CubeCoordinate;
use bevy::prelude::*;
use serde::{Deserialize, Serialize};

#[cfg(feature = "graphics")]
use super::gamemodes::campaign::levels::{
    tutorial::basic_tile_functions::TilesTutorialState,
    tutorial::early_game_tiles::EarlyGameTutorialState, CampaignTutorial,
};

use super::{
    gamemodes::{
        campaign::levels::{
            challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState, CampaignGroups,
        },
        challenge::ChallengeGameState,
        zen::ZenGameState,
        GameMode,
    },
    map::Map,
};

pub const BUILD_PLATE_HEIGHT: f32 = 0.0;
const BUILD_PLATE_INITIAL_SIZE: usize = 4;
const BUILD_PLATE_GROWTH_THRESHOLD: f32 = 0.6;

macro_rules! define_build_area_plugin {
    ($({
        gamemode: $gm:expr,
        transition: $transition:expr$(,)?
    }),*) => {
        pub struct BuildAreaPlugin;
        impl Plugin for BuildAreaPlugin {
            fn build(&self, app: &mut bevy::prelude::App) {
                $(
                    app.add_systems(OnEnter($gm), setup_build_area);
                    app.add_systems(OnExit($gm), despawn_build_area);
                    app.add_systems($transition, grow_build_area);
                )*
            }
        }
    };
}

#[cfg(feature = "graphics")]
define_build_area_plugin![
    {
        gamemode: GameMode::Challenge,
        transition: OnTransition {
            exited: ChallengeGameState::BetweenRounds,
            entered: ChallengeGameState::InRound,
        }
    },
    {
        gamemode: CampaignGroups::CampaignChallenge,
        transition: OnTransition {
            exited: CampaignChallengeState::BetweenRounds,
            entered: CampaignChallengeState::InRound,
        },
    },
    {
        gamemode: CampaignGroups::CampaignWettbewerb,
        transition: OnTransition {
            exited: CampaignWettbewerbState::BetweenRounds,
            entered: CampaignWettbewerbState::InRound,
        },
    },
    {
        gamemode: GameMode::Zen,
        transition: OnTransition {
            exited: ZenGameState::BetweenRounds,
            entered: ZenGameState::InRound,
        }
    },
    {
        gamemode: CampaignTutorial::TilesTutorial,
        transition: OnTransition {
            exited: TilesTutorialState::InGame,
            entered: TilesTutorialState::InGame,
        }
    },
    {
        gamemode: CampaignTutorial::EarlyGameTutorial,
        transition: OnTransition {
            exited: EarlyGameTutorialState::InGame,
            entered: EarlyGameTutorialState::InGame,
        }
    }
];

#[cfg(not(feature = "graphics"))]
define_build_area_plugin![
    {
        gamemode: GameMode::Challenge,
        transition: OnTransition {
            exited: ChallengeGameState::BetweenRounds,
            entered: ChallengeGameState::InRound,
        }
    },
    {
        gamemode: CampaignGroups::CampaignChallenge,
        transition: OnTransition {
            exited: CampaignChallengeState::BetweenRounds,
            entered: CampaignChallengeState::InRound,
        },
    },
    {
        gamemode: CampaignGroups::CampaignWettbewerb,
        transition: OnTransition {
            exited: CampaignWettbewerbState::BetweenRounds,
            entered: CampaignWettbewerbState::InRound,
        },
    },
    {
        gamemode: GameMode::Zen,
        transition: OnTransition {
            exited: ZenGameState::BetweenRounds,
            entered: ZenGameState::InRound,
        }
    }
];

fn grow_build_area(mut build_area: Query<&mut BuildArea>, map: Res<Map>) {
    let Ok(mut build_area) = build_area.get_single_mut() else {
        return;
    };
    // assums corrext implementation of size_hint
    let map_size = build_area
        .center
        .get_area(build_area.radius - 1)
        .size_hint()
        .0;
    if map.len() as f32 / map_size as f32 > BUILD_PLATE_GROWTH_THRESHOLD {
        build_area.radius += 1;
    }
}

#[derive(Component, Debug, Clone, Serialize, Deserialize)]
pub struct BuildArea {
    pub center: CubeCoordinate,
    pub radius: usize,
}
impl BuildArea {
    pub fn check(&self, coord: CubeCoordinate) -> bool {
        self.center.distance(coord) < self.radius
    }
}
impl Default for BuildArea {
    fn default() -> Self {
        Self {
            center: CubeCoordinate::new(0, 0, 0),
            radius: BUILD_PLATE_INITIAL_SIZE,
        }
    }
}

pub fn setup_build_area(mut commands: Commands) {
    commands.spawn(BuildArea::default());
}

fn despawn_build_area(mut commands: Commands, build_areas: Query<Entity, With<BuildArea>>) {
    for build_area in build_areas.iter() {
        commands.entity(build_area).despawn_recursive();
    }
}
