use bevy::prelude::*;
use serde::{Deserialize, Serialize};

use crate::{
    coordinates::CubeCoordinate,
    game::{
        build_area::BuildArea,
        controls::{CreativeInventory, CreativeInventorySlots, PanOrbitCam},
        gamemodes::GameMode,
        map::{RawMap, TileSaveData},
        metrics::{Metrics, MetricsRate, TargetMetrics},
        resources::ConfigureMarketReq,
        rewards::RewardSend,
        round::HandRedrawCost,
        tiles::{
            chameleon_tile::ChangeChameleonTile, rgb_tile::ChangeRgbTileColor,
            sound_tile::ChangeSoundTileData, TileType,
        },
    },
    websocket::WsMessage,
};

use super::creative_bot::CreativeSettings;

macro_rules! bot_events {
    ($name:ident, $kind:ident {$(
        $variant:ident$(
            ( $($args:ty),+ $(,)? )
        )?
    ),* $(,)?}) => {
        #[derive(Debug, Serialize, Deserialize, Event, Clone)]
        #[serde(tag = "t", content = "c")]
        pub enum $name {$(
            $variant$(
                ( $($args),+ )
            )?
        ),*}
        impl $name{
            pub fn to_kind(&self) -> $kind {
                match self{
                    $(
                        $name::$variant { .. } => $kind::$variant
                    ),*
                }
            }
        }
        #[derive(Debug, Serialize, Deserialize, PartialEq, Eq, Hash, Clone, Copy)]
        pub enum $kind {$(
            $variant
        ),*}
    };
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(tag = "t", content = "c")]
pub enum NotInGame {
    Discovery,
    GameRequest,
}

impl From<NotInGame> for WsMessage {
    fn from(value: NotInGame) -> Self {
        serde_json::to_string(&value).expect("Serializing to json should not be a problem.")
    }
}

// todo: not in game answers
#[derive(Debug, Serialize, Deserialize)]
#[serde(tag = "t", content = "c")]
pub enum NotInGameResponse {
    Identity(BotIdentity),
    AcceptGame,
    RejectGame,
}

impl TryFrom<WsMessage> for NotInGameResponse {
    type Error = serde_json::Error;

    fn try_from(value: WsMessage) -> Result<Self, Self::Error> {
        serde_json::from_str(&value)
    }
}

#[derive(Debug, Serialize, Deserialize)]
pub struct BotIdentity {
    pub name: String,
    pub gamemode: GameMode,
    // ...
}

// event stuff

// InGame
bot_events! { BotEvent, BotEventKind {
    GameOver,
    // challenge relevant
    ChallengeTick(ChallengeTick),
    // other
    TileClicked(TileClickedEvent),
    TileStatus(TileStatusEvent),
    InvChanged(CreativeInventory),
    CamMoved(PanOrbitCam),
    // ...
}}

impl BotEvent {
    pub fn needs_no_registration(&self) -> bool {
        #[allow(clippy::match_like_matches_macro)]
        match self {
            BotEvent::GameOver => true,
            BotEvent::TileStatus(data) => matches!(data.kind, TileStatusKind::Manual),
            _ => false,
        }
    }
}

impl From<BotEvent> for WsMessage {
    fn from(value: BotEvent) -> Self {
        serde_json::to_string(&value).expect("Serialize failed, but this should not possible.")
    }
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct ChallengeTick {
    pub actions_count: Option<usize>,
    pub round: usize,
    pub round_time: f32,
    pub map: RawMap,
    pub hand: Vec<TileType>,
    pub build_area: BuildArea,
    pub redraw: HandRedrawCost,
    pub redraw_time: f32,
    pub rewards: Vec<RewardSend>,
    pub metrics: Metrics,
    pub metrics_target: TargetMetrics,
    pub metrics_rate: MetricsRate,
}

#[derive(Debug, Serialize, Deserialize, Clone)]
pub struct TileClickedEvent {
    coord: CubeCoordinate,
    tile: Option<TileType>,
}

impl TileClickedEvent {
    pub fn new(coord: CubeCoordinate, tile: Option<TileType>) -> Self {
        Self { coord, tile }
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct TileStatusEvent {
    coord: CubeCoordinate,
    kind: TileStatusKind,
    #[serde(skip_serializing_if = "Option::is_none")]
    #[serde(default)]
    data: Option<TileSaveData>,
}

impl TileStatusEvent {
    pub fn new(coord: CubeCoordinate, kind: TileStatusKind, data: Option<TileSaveData>) -> Self {
        Self { coord, kind, data }
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub enum TileStatusKind {
    Manual,
    Added,
    Removed,
    Changed,
}

// action stuff
#[derive(Debug, Serialize, Deserialize)]
#[serde(tag = "t", content = "c")]
pub enum BotAction {
    RegisterEvent(BotEventKind),
    UnRegisterEvent(BotEventKind),
    UnRegisterAllEvents,

    SendNotification(String),

    PlaceTile(TileSaveData),
    TakeTile(CubeCoordinate),
    CollectReward(CubeCoordinate),
    Redraw,
    ConfigureMarketplaceTile(ConfigureMarketReq),
    // creative stuff:
    ConfigureRgbTile(ChangeRgbTileColor),
    ConfigureChameleonTile(ChangeChameleonTile),
    ConfigureSoundTile(ChangeSoundTileData),
    GetTileStatus(CubeCoordinate),
    CreativeSettings(CreativeSettings),
    SetCam(PanOrbitCam),
    SetInv(CreativeInventorySlots),
}

impl BotAction {
    /// returns if the action limit is applied to a given Action
    pub fn is_restricted(&self) -> bool {
        #[allow(clippy::match_like_matches_macro)]
        match self {
            BotAction::RegisterEvent(_)
            | BotAction::UnRegisterEvent(_)
            | BotAction::UnRegisterAllEvents
            | BotAction::SendNotification(_) => false,
            _ => true,
        }
    }
}

#[derive(Debug, Serialize, Deserialize, Deref, DerefMut)]
pub struct ActionList(Vec<BotAction>);

impl TryFrom<WsMessage> for ActionList {
    type Error = serde_json::Error;

    fn try_from(value: WsMessage) -> Result<Self, Self::Error> {
        serde_json::from_str(&value)
    }
}
