use bevy::prelude::*;
use http::Uri;
use serde::{Deserialize, Serialize};

#[cfg(feature = "graphics")]
use crate::game::{
    bot::data::TileClickedEvent,
    controls::{placer::TileClickEvent, CamMovedEvent, CreativeInventory, CreativeInventorySlots},
};
use crate::{
    coordinates::CubeCoordinate,
    game::{
        bot::{
            data::{BotEvent, TileStatusEvent, TileStatusKind},
            BotConnecting, BotConnectingState, BotSettings, BotState, ConnectingTimeout,
        },
        controls::PanOrbitCam,
        gamemodes::{creative::CreativeGameState, GameMode},
        map::{Map, TileSaveData},
        resources::{MarketConfiguredEvent, TilePlacedEvent, TileReclaimedEvent},
        tiles::{
            chameleon_tile::ChameleonTileChangedEvent, rgb_tile::RgbTileChangedEvent,
            sound_tile::SoundTileChangedEvent, ConsistentTileData, TileType,
        },
        DisplayMode, GameConfig,
    },
    prelude::*,
    websocket::WebSocketConnector,
};

macro_rules! define_creative_bot_plugin {
    ($({
        // gamemode to handle
        gamemode: $gm:expr,
        // state in which creative bot should be generated
        state: $state:expr,
    }),*) => {
        pub struct CreativeBotPlugin{
            pub mode: DisplayMode,
        }
        impl Plugin for CreativeBotPlugin {
            fn build(&self, app: &mut App) {
                #[cfg(feature = "graphics")]
                if let DisplayMode::Graphic = self.mode {
                    app.add_systems(
                        Update,
                        tile_click_event
                            .run_if(on_event::<TileClickEvent>)
                            .run_if(never()$(.or(still_in_state($state)))*)
                    );
                    app.add_systems(
                        Update,
                        cam_moved_event
                            .run_if(on_event::<CamMovedEvent>)
                            .run_if(never()$(.or(still_in_state($state)))*)
                    );
                    app.add_systems(
                        Update,
                        move_cam_action
                            .run_if(on_event::<CamMoveReq>)
                            .run_if(never()$(.or(still_in_state($state)))*)
                    );
                    app.add_systems(
                        Update,
                        inv_changed_event
                            .run_if(resource_changed::<CreativeInventory>)
                            .run_if(never()$(.or(still_in_state($state)))*)
                    );
                    app.add_systems(
                        Update,
                        change_inv_slots
                            .run_if(on_event::<CreativeInventorySlots>)
                            .run_if(resource_exists::<CreativeInventory>)
                            .run_if(never()$(.or(still_in_state($state)))*)
                    );
                }
                app.add_event::<CamMoveReq>();
                app.add_event::<TileStatusReq>();
                app.add_systems(
                    Update,
                    tile_status_event.run_if(never()$(.or(still_in_state($state)))*)
                );
                app.add_event::<CreativeSettings>();
                $(
                    app.add_systems(OnEnter($gm), |mut commands: Commands| {
                        commands.insert_resource(CreativeSettings::default());
                    });
                    app.add_systems(OnExit($gm), |mut commands: Commands| {
                        commands.remove_resource::<CreativeSettings>();
                    });
                )*
                app.add_systems(
                    Update,
                    update_creative_settings.run_if(never()$(.or(still_in_state($state)))*)
                );
            }
        }
    };
}

define_creative_bot_plugin![
    {
        gamemode: GameMode::Creative,
        state: CreativeGameState::InGame,
    }
];

pub fn start_creative_bot_game(
    mut commands: Commands,
    mut settings: ResMut<BotSettings>,
    mut bot_state: ResMut<BotState>,
    mut state: ResMut<BotConnectingState>,
    config: Res<GameConfig>,
    mut timer: ResMut<ConnectingTimeout>,
) {
    let Ok(uri) = Uri::builder()
        .scheme("ws")
        .authority(config.bot_url.clone())
        .path_and_query("/")
        .build()
    else {
        warn!("Parsing Bot url failed.");
        *state = BotConnectingState::UrlError;
        return;
    };

    let Ok(connector) = WebSocketConnector::new(uri) else {
        *state = BotConnectingState::UrlError;
        warn!("Calling Bot url failed.");
        return;
    };

    commands.insert_resource(BotConnecting(connector));

    *state = BotConnectingState::WaitingWs;
    settings.action_limit = None;
    settings.disable_on_action = false;
    bot_state.execute_actions = true;
    bot_state.registered_events.clear();
    timer.reset();
}

#[cfg(feature = "graphics")]
pub fn tile_click_event(
    mut bot_event: EventWriter<BotEvent>,
    mut tile_click_event: EventReader<TileClickEvent>,
    map: Res<Map>,
    tiles: Query<&TileType>,
) {
    for event in tile_click_event.read() {
        bot_event.send(BotEvent::TileClicked(TileClickedEvent::new(
            event.tile_pos,
            map.get(event.tile_pos)
                .and_then(|id| tiles.get(id).ok())
                .cloned(),
        )));
    }
}

#[cfg(feature = "graphics")]
pub fn cam_moved_event(
    mut bot_event: EventWriter<BotEvent>,
    mut cam_moved_evt: EventReader<CamMovedEvent>,
    cam_query: Query<&PanOrbitCam>,
) {
    let Ok(cam) = cam_query.get_single() else {
        return;
    };
    if !cam_moved_evt.is_empty() {
        cam_moved_evt.clear();
        bot_event.send(BotEvent::CamMoved(*cam));
    }
}

#[cfg(feature = "graphics")]
pub fn inv_changed_event(mut bot_event: EventWriter<BotEvent>, inv: Res<CreativeInventory>) {
    bot_event.send(BotEvent::InvChanged(inv.clone()));
}

#[derive(Event, Deref)]
pub struct CamMoveReq(pub PanOrbitCam);

#[cfg(feature = "graphics")]
pub fn move_cam_action(
    mut cam_move: EventReader<CamMoveReq>,
    mut cam_query: Query<&mut PanOrbitCam>,
) {
    let Some(mut cam) = cam_query.iter_mut().next() else {
        return;
    };
    let Some(cam_pos) = cam_move.read().last() else {
        return;
    };
    *cam = **cam_pos;
}

#[cfg(feature = "graphics")]
pub fn change_inv_slots(
    mut inv_change: EventReader<CreativeInventorySlots>,
    mut inv: ResMut<CreativeInventory>,
) {
    use crate::game::gamemodes::creative::CREATIVE_INV_MAX_SLOT_COUNT;

    let Some(new_inv) = inv_change.read().last() else {
        return;
    };
    inv.slots = CreativeInventorySlots(
        new_inv
            .iter()
            .take(CREATIVE_INV_MAX_SLOT_COUNT)
            .cloned()
            .collect(),
    );
    if inv.slots.is_empty() {
        inv.slots.push(None);
    }
    inv.selected = inv.selected.clamp(0, inv.slots.len() - 1);
}

#[derive(Debug, Event)]
pub struct TileStatusReq(pub CubeCoordinate);

#[allow(clippy::too_many_arguments)]
pub fn tile_status_event(
    mut bot_event: EventWriter<BotEvent>,
    mut added: EventReader<TilePlacedEvent>,
    mut removed: EventReader<TileReclaimedEvent>,
    mut market: EventReader<MarketConfiguredEvent>,
    mut rgb: EventReader<RgbTileChangedEvent>,
    mut chameleon: EventReader<ChameleonTileChangedEvent>,
    mut sound: EventReader<SoundTileChangedEvent>,
    mut tile_stat: EventReader<TileStatusReq>,
    tiles: Query<ConsistentTileData, With<TileType>>,
    map: Res<Map>,
) {
    for add in added.read() {
        bot_event.send(BotEvent::TileStatus(TileStatusEvent::new(
            add.0.coord,
            TileStatusKind::Added,
            Some(add.0.clone()),
        )));
    }

    for remove in removed.read() {
        bot_event.send(BotEvent::TileStatus(TileStatusEvent::new(
            remove.0,
            TileStatusKind::Removed,
            None,
        )));
    }

    for (coord, kind) in market
        .read()
        .map(|market| (market.coord, TileStatusKind::Changed))
        .chain(rgb.read().map(|evt| (evt.0, TileStatusKind::Changed)))
        .chain(chameleon.read().map(|evt| (evt.0, TileStatusKind::Changed)))
        .chain(sound.read().map(|evt| (evt.0, TileStatusKind::Changed)))
        .chain(tile_stat.read().map(|evt| (evt.0, TileStatusKind::Manual)))
    {
        bot_event.send(BotEvent::TileStatus(TileStatusEvent::new(
            coord,
            kind,
            map.get(coord)
                .and_then(|id| tiles.get(id).ok())
                .map(|data| TileSaveData {
                    coord: *data.0,
                    tile: *data.1,
                    data: TileType::serialize_tile_data(data),
                    reclaim: data.2.map(|f| f.timer.remaining_secs()),
                }),
        )));
    }
}

#[derive(Debug, Clone, Event, Resource, Serialize, Deserialize)]
pub struct CreativeSettings {
    /// allows user to move the cam origin
    pub move_cam: bool,
    /// allows user to rotate the cam around the origin
    pub rotate_cam: bool,
    /// allows user to zoom the cam
    pub zoom_cam: bool,
    /// allows user to open tile info / settings popup on user click
    pub tile_popup: bool,
    /// allows user to place tiles
    pub place_tiles: bool,
    /// allows user to remove tiles
    pub remove_tiles: bool,
    /// allow user to open inventory and change inventory content
    pub change_inv: bool,
    /// show inventory slots
    pub show_inv_slots: bool,
}

impl Default for CreativeSettings {
    fn default() -> Self {
        Self {
            move_cam: true,
            rotate_cam: true,
            zoom_cam: true,
            tile_popup: true,
            place_tiles: true,
            remove_tiles: true,
            change_inv: true,
            show_inv_slots: true,
        }
    }
}

fn update_creative_settings(
    mut settings: ResMut<CreativeSettings>,
    mut update: EventReader<CreativeSettings>,
) {
    if let Some(update) = update.read().last() {
        *settings = update.clone();
    }
}

pub mod creative_settings_cond {
    use super::CreativeSettings;
    use bevy::prelude::*;

    // to use in a system

    /// Checks if user is allowed to move the cam. Using a ref.
    pub fn move_cam_ref(cr_settings: &Option<Res<CreativeSettings>>) -> bool {
        cr_settings.as_ref().is_none_or(|cr_s| cr_s.move_cam)
    }

    /// Checks if user is allowed to rotate the cam. Using a ref.
    pub fn rotate_cam_ref(cr_settings: &Option<Res<CreativeSettings>>) -> bool {
        cr_settings.as_ref().is_none_or(|cr_s| cr_s.rotate_cam)
    }

    /// Checks if user is allowed to zoom the cam. Using a ref.
    pub fn zoom_cam_ref(cr_settings: &Option<Res<CreativeSettings>>) -> bool {
        cr_settings.as_ref().is_none_or(|cr_s| cr_s.zoom_cam)
    }

    /// Checks if user is allowed to place tiles. Using a ref.
    pub fn place_tiles_ref(cr_settings: &Option<Res<CreativeSettings>>) -> bool {
        cr_settings.as_ref().is_none_or(|cr_s| cr_s.place_tiles)
    }

    /// Checks if user is allowed to remove tiles. Using a ref.
    pub fn remove_tiles_ref(cr_settings: &Option<Res<CreativeSettings>>) -> bool {
        cr_settings.as_ref().is_none_or(|cr_s| cr_s.remove_tiles)
    }

    /// Checks if user is allowed to change the creative inventory / is allowed to open it. Using a ref.
    pub fn change_inv_ref(cr_settings: &Option<Res<CreativeSettings>>) -> bool {
        cr_settings.as_ref().is_none_or(|cr_s| cr_s.change_inv)
    }

    // to use as run condition
    /// Run condition: checks if user is allowed to open tile popup on click
    pub fn tile_popup(cr_settings: Option<Res<CreativeSettings>>) -> bool {
        cr_settings.is_none_or(|cr_s| cr_s.tile_popup)
    }

    /// Run condition: Checks if user is allowed to change the inventory / is allowed to open it.
    pub fn change_inv(cr_settings: Option<Res<CreativeSettings>>) -> bool {
        cr_settings.is_none_or(|cr_s| cr_s.change_inv)
    }

    /// Run condition: Checks if inventory slots should be shown.
    pub fn show_inv_slots(cr_settings: Option<Res<CreativeSettings>>) -> bool {
        cr_settings.is_none_or(|cr_s| cr_s.show_inv_slots)
    }
}
