use bevy::{asset::LoadState, audio::Volume, ecs::system::EntityCommand, prelude::*};
use std::collections::HashMap;

use crate::{
    errors::ErrorDisplay,
    game::{
        asset_loading::{AssetLoadStateDump, LACManager, LoadableAssetCollection},
        controls::KeyboardEvent,
        resources::{TilePlacedEvent, TileReclaimedEvent},
        round::{FreeRedraw, GameOver, PhaseEnd, Redraw},
    },
    prelude::Translate,
    settings::ActiveSettingsBank,
    ui::{commons::ScreenshotEvent, UiClickEvent},
    AppState,
};

pub struct SoundEffectsPlugin;
impl Plugin for SoundEffectsPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.register_asset_collection::<SoundEffectAssets>();
        // NOTE: only required if the round system is running
        app.add_systems(
            Update,
            round_sound_effects.run_if(in_state(AppState::InGame)),
        );
        // NOTE: only requied if the placer is running
        app.add_systems(
            Update,
            modifiy_sound_effects.run_if(in_state(AppState::InGame)),
        );
        app.add_systems(Update, commons_effects);
    }
}

#[derive(Debug, Clone)]
#[allow(dead_code)]
pub enum SoundEffects {
    FreeRedraw,
    Redraw,
    Placing,
    GameOver,
    PhaseEnd,
    Screenshot,
    Click,
}

impl SoundEffects {
    /// Returns a list of all available sound effects
    fn list() -> [Self; 7] {
        [
            Self::FreeRedraw,
            Self::Redraw,
            Self::Placing,
            Self::GameOver,
            Self::PhaseEnd,
            Self::Screenshot,
            Self::Click,
        ]
    }
    /// Returns Vec of path and adjusted volume for a SoundEffect.
    fn get_audio_paths(&self) -> Vec<(&'static str, f32)> {
        match self {
            SoundEffects::FreeRedraw => vec![("sounds/bookFlip3.ogg", 1.0)],
            SoundEffects::Redraw => vec![
                ("sounds/cloth1.ogg", 0.75),
                ("sounds/cloth2.ogg", 0.75),
                ("sounds/cloth3.ogg", 0.75),
                ("sounds/cloth4.ogg", 0.75),
            ],
            SoundEffects::Placing => vec![("sounds/bong_001.ogg", 1.0)],
            SoundEffects::GameOver => vec![("sounds/error_007.ogg", 1.0)],
            SoundEffects::PhaseEnd => vec![("sounds/drop_003.ogg", 1.0)],
            SoundEffects::Screenshot => vec![("sounds/camera-shutter.oga", 1.0)],
            SoundEffects::Click => vec![("sounds/click_002.ogg", 0.5)],
        }
    }

    /// Returns a random path and adjusted volume for a SoundEffect.
    fn get_random(&self) -> (&'static str, f32) {
        let paths = self.get_audio_paths();
        let index = fastrand::usize(0..paths.len());
        paths[index]
    }
}

fn modifiy_sound_effects(
    mut commands: Commands,
    mut event_tile_placed: EventReader<TilePlacedEvent>,
    mut event_tile_reclaimed: EventReader<TileReclaimedEvent>,
) {
    for _ in event_tile_placed.read() {
        commands
            .spawn_empty()
            .queue(PlaySoundEffect::new(SoundEffects::Placing));
    }
    for _ in event_tile_reclaimed.read() {
        // TODO: use different sound!
        commands
            .spawn_empty()
            .queue(PlaySoundEffect::new(SoundEffects::Placing));
    }
}

fn round_sound_effects(
    mut commands: Commands,
    mut event_free_redraw: EventReader<FreeRedraw>,
    mut event_redraw: EventReader<Redraw>,
    mut event_game_over: EventReader<GameOver>,
    mut event_phase_end: EventReader<PhaseEnd>,
) {
    for _ in event_free_redraw.read() {
        commands
            .spawn_empty()
            .queue(PlaySoundEffect::new(SoundEffects::FreeRedraw));
    }
    for _ in event_redraw.read() {
        commands
            .spawn_empty()
            .queue(PlaySoundEffect::new(SoundEffects::Redraw));
    }
    for _ in event_game_over.read() {
        commands
            .spawn_empty()
            .queue(PlaySoundEffect::new(SoundEffects::GameOver));
    }
    for _ in event_phase_end.read() {
        commands
            .spawn_empty()
            .queue(PlaySoundEffect::new(SoundEffects::PhaseEnd));
    }
}

fn commons_effects(
    mut commands: Commands,
    mut event_screenshot: EventReader<ScreenshotEvent>,
    mut event_ui_click: EventReader<UiClickEvent>,
    mut event_kb: EventReader<KeyboardEvent>,
) {
    for _ in event_screenshot.read() {
        commands
            .spawn_empty()
            .queue(PlaySoundEffect::new(SoundEffects::Screenshot));
    }
    for _ in event_ui_click.read() {
        commands
            .spawn_empty()
            .queue(PlaySoundEffect::new(SoundEffects::Click));
    }
    for _ in event_kb.read() {
        commands
            .spawn_empty()
            .queue(PlaySoundEffect::new(SoundEffects::Click));
    }
}

pub struct PlaySoundEffect {
    sound: SoundEffects,
    volume_factor: f32,
}
impl PlaySoundEffect {
    pub fn new(sound: SoundEffects) -> Self {
        Self {
            sound,
            volume_factor: 1.0,
        }
    }

    #[allow(dead_code)]
    pub fn with_volume(mut self, volume_factor: f32) -> Self {
        self.volume_factor = volume_factor;
        self
    }
}
impl EntityCommand for PlaySoundEffect {
    fn apply(self, id: Entity, world: &mut World) {
        let scene = {
            let assets = world
                .get_resource::<SoundEffectAssets>()
                .expect("The LoadingScreen should have started loading the assets at this point.");
            let sound = self.sound.get_random();
            let path = sound.0;
            let handle = assets
                .0
                .get(path)
                .expect("The LoadingScreen should have ensured that the asset was loaded");
            (handle.clone_weak(), sound.1)
        };
        let volume = {
            world
                .get_resource::<ActiveSettingsBank>()
                .map_or(1.0, |settings| settings.audio.get_sfx_percent())
        };

        world.entity_mut(id).insert((
            AudioPlayer(scene.0),
            PlaybackSettings::DESPAWN
                .with_volume(Volume::new(volume * self.volume_factor * scene.1)),
        ));
    }
}

#[derive(Resource, Default)]
struct SoundEffectAssets(HashMap<&'static str, Handle<AudioSource>>);
impl LoadableAssetCollection for SoundEffectAssets {
    fn load_all(asset_server: &AssetServer) -> Self {
        let mut collection = SoundEffectAssets(HashMap::new());
        for effect in SoundEffects::list() {
            let files = effect.get_audio_paths();
            for file in files {
                if collection.0.contains_key(file.0) {
                    // the handle is also used by a different sound effect
                    // and has already been loaded
                    continue;
                }
                collection.0.insert(file.0, asset_server.load(file.0));
            }
        }
        collection
    }

    fn check_all(
        &mut self,
        asset_server: &AssetServer,
    ) -> crate::game::asset_loading::AssetLoadStateDump {
        let mut dump = AssetLoadStateDump::default();

        for handle in self.0.values() {
            dump.requested += 1;
            match asset_server.get_load_state(handle) {
                Some(LoadState::Loaded) => dump.loaded += 1,
                Some(LoadState::Failed(_)) => dump.failed += 1,
                _ => (),
            }
        }

        dump
    }

    fn get_error(
        &self,
        localization: Option<&crate::i18n::Localization>,
    ) -> Option<crate::errors::ErrorDisplay> {
        Some(ErrorDisplay {
            title: localization
                .and_then(|l| l.try_translate("missing-sound-effects-title"))
                .unwrap_or(String::from("Some Sound Effects are missing")),
            description: localization
                .and_then(|l| l.try_translate("missing-sound-effects-description"))
                .unwrap_or(String::from("Sound Effects provide auditory feedback.")),
            link: None,
        })
    }
}
