# 📌 States

States are a powerful bevy tool, which can be used for scheduling,
run-conditions and more.

To define a new *State*, you can simply derive 
[`States`](https://docs.rs/bevy/latest/bevy/prelude/trait.States.html),
`rust-analyzer` will prompt you to add the additional `derive` attributes.

```rust
#[derive(States, Debug, Clone, Copy, PartialEq, Eq, Hash)]
pub enum AppState {
}
```

Afterwards, you can use `app.insert_state` or `app.init_state` 
to register the state with the app.

If you want to change the state value,
simply request it using the `NextState` resource.
```rust
fn start_game(mut next_state: ResMut<NextState<AppState>>) {
    next_state.set(AppState::InGame);
}
```

When reading the state, simply use the `State` resource
```rust
fn get_game(state: Res<State<AppState>>) {
    match state.get() {
        [...]
    }
}
```

## 📍 SubStates

When adding new game functionality, 
it is recommended, to hook into the existing state system, by using
[`SubStates`](https://docs.rs/bevy/latest/bevy/prelude/trait.SubStates.html).

By using *SubStates*, we can guarantee that the substate only exists,
whenever the parent state also exists. 
Allowing for more complex logic arrangements, 
without having to check other states or running cleanup rules.

You can define *SubStates* by deriving `Substates` 
and setting the `source` attribute.
(`rust-analyzer` will prompt you to add the additional `derive` attributes)
```rust
#[derive(SubStates, Default, Debug, Hash, PartialEq, Eq, Clone)]
#[source(AppState = AppState::AppMenu)]
pub enum AppMenuState {
    [...]
}
```
In this case `AppState` is the parent state,
and the `AppMenuState` only exists whenever
`AppState` is in the `AppState::AppMenu` state.

You can use `NextState` and `State` when using *SubStates*,
the same you would when working with *States*, 
however, keep in mind that the state only exists if the parent state
is in the correct state - so you might have to wrap them in an `Option`.

To set the default value of a substate, 
when changing the parent state, simply update both at the same time
```rust
fn launch_settings(
    mut app_state: ResMut<NextState<AppState>>
    mut menu_state: ResMut<NextState<AppMenuState>>
    ) {
    app_state.set(AppState::AppMenu);
    menu_state.set(AppMenuState::SettingsMenu);
}
```

To learn more, have a look at the
[bevy docs](https://docs.rs/bevy/latest/bevy/prelude/trait.SubStates.html).

## 📈 Overview

```mermaid
flowchart LR;

AS[AppState]
AS_LA[LoadApp]
AS_AM[AppMenu]
AS_IG[InGame]
AS_E[Error]

AS --- AS_LA
AS --- AS_AM
AS --- AS_GC
AS --- AS_IG
AS --- AS_E

AMS[AppMenuState]
AMS_MM[MainMenu]
AMS_SM[SettingsMenu]
AMS_SG[SetupMenu]
AMS_A[About]

AS_AM --substate--> AMS
AMS --- AMS_MM
AMS --- AMS_SM
AMS --- AMS_SG
AMS --- AMS_A

SMS[SetupMenuState]
SMS_GS[GameModeSelect]
SMS_CGS[CampaignGroupSelect]
SMS_CLS[CampaignLevelSelect]
SMS_CG[ConfigureGame]

SMS --- SMS_GS
SMS --- SMS_CGS
SMS --- SMS_CLS
SMS --- SMS_CG

PS[GamePauseState]
PS_R[Running]
PS_P[Paused]
AS_IG --substate--> PS
PS --- PS_R
PS --- PS_P

PUS[PauseUiState]
PUS_H[Hidden]
PUS_V[Visibile]
PS_P --substate-->PUS
PUS --- PUS_H
PUS --- PUS_V

PSet[PauseSettingsState]
PSet_H[Hidden]
PSet_V[Visibile]
PS_P --substate-->PSet
PSet --- PSet_H
PSet --- PSet_V

GM[GameMode]
GM_CP[Challange]
GM_CR[Creative]
GM_CM[Campaign]
GM_Z[Zen]

AS_IG --substate--> GM
GM --- GM_CP
GM --- GM_CR
GM --- GM_CM
GM --- GM_Z

CPM[ChallengeGameState]
CPM_IR[InRound]
CPM_BR[BetweenRounds]
CPM_GO[GameOver]
CPM_SP[Spectate]

GM_CP --substate--> CPM
CPM --- CPM_IR
CPM --- CPM_BR
CPM --- CPM_GO
CPM --- CPM_SP

SVS[StatisticsViewState]
SVS_H[Hidden]
SVS_FG[FromGame]
SVS_FP[FromPaused]
PS_P --substate--> SVS
SVS --- SVS_H
SVS --- SVS_FG
SVS --- SVS_FP

CRM[CreativeGameMode]
CRM_I[InGame]
GM_CR --substate--> CRM
CRM --- CRM_I
```
