# ⏸ Pause API

When the player is in-game, we create a state called `GamePauseState`.
If the game is running, it has the value `Running`,
this can be used as a run-condition for systems that should only run
if the game is running
```rust
.run_if(in_state(GamePauseState::Running))
```

The game can be paused, by setting the `GamePauseState` to `Paused`.
This way, the Virtual time won't continue running.

If TTE is running in `Graphic` mode,
the camera view will always be blurred in the `Paused` state.

## 🧰 Add your own interactions

To add custom interactions, that require the game to be paused,
simply define a new state as a substate of `GamePauseState::Paused`.
Remember to set the *Hidden* state as the default,
to prevent your screen from accidentally showing up if a different system
pauses the game
```rust
#[derive(SubStates, Default, Clone, Hash, PartialEq, Eq, Debug)]
#[source(GamePauseState = GamePauseState::Paused)]
pub enum DemoPauseState {
    #[default]
    Hidden,
    Shown
}
```

Now you can use `.run_if(in_state(DemoPauseState::Shown))` as a run-condition
for your UI system.

Because `Hidden` is the default variant,
you have to request both `ResMut<NextState<GamePauseState>>`
and `ResMut<NextState<DemoPauseState>>`.
```rust
fn toggle(
    mut pause: ResMut<NextState<GamePauseState>>,
    mut demo: ResMut<NextState<DemoPauseState>>
) {
    pause.set(GamePauseState::Paused);
    demo.set(DemoPauseState::Shown);
}
```
Note that because we are using sub-states,
you only have to set the `GamePauseState` to `Running` to 
dismiss the pause screen and continue the game.

## 📈 Diagram

```mermaid
flowchart TD;
AS[AppState]
AS_IG[InGame]
AS --- AS_IG

GPS[GamePauseState]
GPS_R[Running]
GPS_P[Paused]
GPS --- GPS_R
GPS --- GPS_P
AS_IG --substate--> GPS

PUIS[PauseUiState]
PUIS_S[Shown]
PUIS_H[Hidden]
PUIS --- PUIS_S
PUIS --- PUIS_H
GPS_P --substate--> PUIS

SVS[StatisticsViewState]
SVS_H[Hidden]
SVS_P[FromPaused]
SVS_G[FromGame]
SVS --- SVS_G
SVS --- SVS_P
SVS --- SVS_H
GPS_P --substate--> SVS
```
