# 🎮 Game-modes

A game mode defines how a player may interact with the game / world. \
Different game modes may include
- different ways of interaction
- different goals
- different tiles
- different understandings of time and rounds

The `GameMode` (sub)state reflects the current game mode 
and is only available if the player is in the `AppState::InGame`.

## 🧰 Creating new game-modes

First of all,
you have to add the new game mode to the `GameMode` *enum*.
For example:
```rust
/// [...]
pub enum GameMode {
    // [...]
    /// Briefly explain your gamemode as a doc comment
    MyGamemode
}
```

Second of all, you have to create a `mygamemode.rs` file in the `gamemodes`
directory. \
To be able to start a game in the game mode, you have to write a one-shot system,
which can be called to start the game.
```rust
pub fn init_mygamemode_mode(
    _config: Res<MyGamemodeModeConfig>,
    mut app_state: ResMut<NextState<AppState>>,
    mut gamemode: ResMut<NextState<GameMode>>,
) {
    app_state.set(AppState::InGame);
    gamemode.set(GameMode::MyGamemode);
    // TODO: apply config data
}
```
This system has to be added to the `commands.queue` call in `src/ui/setup_menu.rs`.

If your game mode requires additional configuration (i.e., a seed) or custom
substates, you should define them in this file as well.
When defining custom substates, you most likely want to use the `GameMode`
variant you created earlier as a parent state, to ensure that your game mode 
only exists when the correct game mode is selected.
```rust
#[derive(Debug, Clone, Copy, Default, PartialEq, Eq, Hash, SubStates)]
#[source(GameMode = GameMode::MyGamemode)]
pub enum MyGamemodeGameState {
    /// the player is currently able to interact with the world
    #[default]
    InGame,
}
```

This file should also contain a `MyGamemodePlugin`, which registers these 
states and resources.

```rust
pub struct MyGamemodeModePlugin {
    pub display_mode: DisplayMode,
}
impl Plugin for MyGamemodeModePlugin {
    fn build(&self, app: &mut App) {
        app.add_sub_state::<MyGamemodeGameState>();
    }
}
```

That's it for the basic game mode setup.
Now all that is left to do, is configure the plugins that make up the 
game mode.

### Reusing existing plugins
If you take a look at one of our existing plugins (i.e., `controls/cam.rs`),
you will find that they follow a similar structure:
1. macro definition
2. macro call

Assuming that you can reuse the plugin for your game mode as is,
you can simply register your plugin in the macro call. \
Let's take a look at the camera plugin:
As you can see when looking at the macro definition or at the way the other 
game modes are registered, every registration has a `states` list and a 
`gamemode`:
```rust
    {
        states: vec![ChallengeGameState::InRound, ChallengeGameState::Spectate],
        gamemode: GameMode::Challenge
    },
```
Similarly, you want to add your game mode to the section by using the substate
and the `gamemode`
```rust
define_camera_plugin![
    // [...]
    {
        states: vec![MyGamemodeGameState::InGame],
        gamemode: GameMode::MyGamemode
    },
];
```

### Creating new plugins
New plugins should be placed in a new file. 
A non-exhaustive list of plugin locations is shown below:
(Feel free to create new modules as you see fit)
- `ui/in_game` - Plugins related to UI interactions
- `game/bot` - Plugins adding bot support to new game modes
- `game/controls` - Plugins adding custom placement or camera systems
- `game/visuals` - Plugins adding visual elements like animations

After creating a new file (i.e `example.rs`) you can initialize it with a base
structure:
1. macro definition (defines the macro and contains the plugin code)
2. macro call (used to register game modes for the plugin)

```rust
macro_rules! define_example_plugin {
    ($({
        // configuration options
    }),*) => {
        pub struct ExamplePlugin;
        impl Plugin for ExamplePlugin {
            fn build(&self, _app: &mut App) {
                // register systems, resources and events
            }
        }
    };
}

define_example_plugin! [
    // register game modes here
]

// define systems, resources and events here
```

To allow game modes to pass configuration options to your plugin,
you most likely want to expose them as macro matchers. \
You can look at the [official rust macro guide][macro-ex] for more information.

```rust
$({
    state: $state:expr,
    gamemode: $gm:expr$(,)?
}),*
```

Every plugin should support being used by multiple game modes.
This is why you have to write macro code depending on the run conditions you 
want:

To run systems `OnEnter` (or similar), you have to generate the system 
registration for every plugin. This can be done using the following expression:
(Assumes a `$gm` configuration entry)
```rust
$(
    app.add_systems(OnExit($gm), despawn_cam);
)*
```

Systems that are registered to run on `Update` often use `run_if(in_state)`,
to allow multiple plugins to use the same registration, you can use the 
following expression:
```rust
app.add_systems(
    Update,
    rotate_cam 
        .run_if(never()$(.or(in_state($state)))*));
```

For more specific usage examples, simply take a look at the already existing
plugins.

[macro-ex]: https://doc.rust-lang.org/reference/macros-by-example.html

## 📈 Overview

```mermaid
flowchart LR;


AS_GC[GameConfig]
AS_IG[InGame]
AS_D[...]
AS[AppState]

AS --- AS_GC
AS --- AS_IG
AS --- AS_D


PS[GamePauseState]
PS_R[Running]
PS_P[Paused]
AS_IG --substate--> PS
PS --- PS_R
PS --- PS_P

PUS[PauseUiState]
PUS_H[Hidden]
PUS_V[Visibile]
PS_P --substate-->PUS
PUS --- PUS_H
PUS --- PUS_V

GM[GameMode]
GM_CP[Challange]
GM_CR[Creative]
GM_CM[Campaign]
GM_Z[Zen]

AS_IG --substate--> GM
GM --- GM_CP
GM --- GM_CR
GM --- GM_CM
GM --- GM_Z

CPM[ChallengeGameState]
CPM_IR[InRound]
CPM_BR[BetweenRounds]
CPM_GO[GameOver]
CPM_SP[Spectate]

GM_CP --substate--> CPM
CPM --- CPM_IR
CPM --- CPM_BR
CPM --- CPM_GO
CPM --- CPM_SP

SVS[StatisticsViewState]
SVS_H[Hidden]
SVS_FG[FromGame]
SVS_FP[FromPaused]
PS_P --substate--> SVS
SVS --- SVS_H
SVS --- SVS_FG
SVS --- SVS_FP

CRM[CreativeGameMode]
CRM_I[InGame]
GM_CR --substate--> CRM
CRM --- CRM_I
```
