# Bot API
TTE has Bot support. That means you can try to write a Bot to beat your highscore.

To write a Bot yourself, you can use one of our Bot libraries (Take a look at our git [organization](https://codeberg.org/terratactician-expandoria)) 
or write one yourself, by implementing the protocol described in this file.

## System Architecture
TTE Bot's communicate over a WebSocket with the game using JSON. 
The Bot has to open a WebSocket Server and the Game has to connect to it.

> This way the game can run in a Browser and still connect to a Bot.

## Protocol
The Protocol is split into two parts:
- Not in game
- In game

Depending on the connection state, only the appropriate packages can be send.

All messages have the following form:
```json
{
    "t": "Message Type",
    "c": // content of the message.
}
```

The `c` key will be left out, if a message has no content.

An exception to this are [Actions](#in-game).
Single actions are encoded like described before.
But Actions are always sent in an array like:
```json
[
    {"t": "ActionType1", "c": /* data */},
    {"t": "ActionType56", "c": /* data */},
    // ...
]
```

In some cases, non-primitive datatypes are used.
They are described [here](#datatypes).

### Not in game
Message types:

#### `Discovery`
- direction: Game to Bot
- description: Discovery request, to request the [`Identity`](#identity) response, by the Bot. May be used to validate a connection. This Request has no effect on the Bot.
- content: none  

#### `GameRequest`
- direction: Game to Bot
- description: Requests the Bot to start a game. The Bot will acknowledge this request with [`AcceptGame`](#acceptgame) or [`RejectGame`](#rejectgame).
- content: none

#### `Identity`
- direction: Bot to Game
- description: Acknowledgement of [`Discovery`](#discovery). Provides the game with the only supported game mode by the bot. 
- content:
```json
{
    "name": string, 
    "gamemode": "Challenge" | "Creative",
}
```

#### `AcceptGame`
- direction: Bot to Game
- description: The bot accepts the [`GameRequest`](#gamerequest).
- content: none

#### `RejectGame`
- direction: Bot to Game
- description: The bot rejects the [`GameRequest`](#gamerequest).
- content: none

### In game
The messages, in game, are called:
- Action: Message from the Bot to the Game
- Event: Message from the Game to the Bot

Not all Events and Actions are valid in all gamemodes.

Some Actions are restricted in some gamemodes. 
In challenge mode, that means they can only be sent in the fist ActionList package after a [`ChallengeTick`](#event-challengetick) Event.

#### Event: `ChallengeTick`
- gamemode: Challenge
- description: Request by the Game for the Bot to make a move. The Bot is only allowed to send an action list after this Event once.
- content: 
```json
{
    "actions_count": uint | null, // the max amount of actions in the next ActionList or unlimited.
    "round": uint, // round number
    "round_time": float, // remaining time in the current round in sec.
    "map": [Tile, /* ... */],
    "hand": [TileType, /* ... */], // available cards in the players hand.
    "build_area": { // the build area the game is restricted to.
        "center": CubeCoordinate,
        "radius": uint,
    },
    "redraw": null | Metrics, // redraw costs or null if free.
    "redraw_time": float, // time until free redraw in sec.
    "rewards": [{ // available rewards
        "val": float, // abstract value (in 0..1)
        "kind": RewardType,
        "coord": CubeCoordinate,
    }, /* ... */ ],
    "metrics": Metrics, // the current metrics
    "metrics_target": Metrics, // the current metrics goal
    "metrics_rate": Metrics, // the current metrics rate
}
```

#### Event: `GameOver`
- gamemode: all
- description: Indicates that a game is over. A disconnection implies this event. (It does not need to be send.)  
- content: none

#### Event: `TileClicked`
- gamemode: creative
- description: Will be send if the player clicks on a placed tile. The event needs to be registered.
- content: 
```json
{
    "tile": TileType,
    "coord": CubeCoordinate,
}
```

#### Event: `TileStatus`
- gamemode: creative
- description: The Bot can listen to this event to acquire map updates. The event will be send on action: `GetTileStatus` or on map events (added, changed, removed). This event may acknowledge bot actions. 
- content: 
```json
{
    "coord": CubeCoordinate,
    "kind": "Manual" | "Added" | "Removed" | "Changed",
    "data": Tile | null, // if null entry maybe skipped
}
```

#### Event: `InvChanged`
- gamemode: creative
- description: The Bot can listen on this event to acquire inventory changes (content or selected slot). This event may acknowledge bot actions.
- content: 
```json
{
    "selected": uint,
    "slots": [ TileType | null, /* ... */]
}
```

#### Event: `CamMoved`
- gamemode: creative
- description: The Bot can listen on this event to receive camera position updates. This event may acknowledge bot actions.
- content: 
```json
{
    "focus": { "x": float, "y": float, "z": float },
    "orbit": { "x": float, "y": float },
    "radius": float,
}
```

#### Action: `RegisterEvent`
- gamemode: all
- restriction: never restricted
- description: Request that the given event will be sent and not ignored.
- content: the [`BotEventKind`](#boteventkind) string

#### Action: `UnRegisterEvent`
- gamemode: all
- restriction: never restricted
- description: Reverse the effect of `RegisterEvent`.
- content: the [`BotEventKind`](#boteventkind) string

#### Action: `UnRegisterAllEvents`
- gamemode: all
- restriction: never restricted
- description: Reverse the effect of all `RegisterEvent` calls.
- content: none

#### Action: `SendNotification`
- gamemode: all
- restriction: never restricted
- description: Sends notification to the player.
- content: `String`

#### Action: `Redraw`
- gamemode: challenge
- restriction: restricted in challenge mode
- description: Request the game to redraw a deck, if possible.
- content: none

#### Action: `PlaceTile`
- gamemode: all
- restriction: restricted in challenge mode
- description: Request the game to place a tile. In challenge mode the card has to be in the hand.
- content: [Tile](#tile)

#### Action: `TakeTile`
- gamemode: all
- restriction: restricted in challenge mode
- description: Request the game to remove a tile from the map. In challenge mode a slot in the hand must be free.
- content: `CubeCoordinate`

#### Action: `CollectReward`
- gamemode: challenge
- restriction: restricted in challenge mode
- description: Request the game to collect a reward, to earn its value.
- content: `CubeCoordinate`

#### Action: `ConfigureMarketplaceTile`
- gamemode: challenge
- restriction: restricted in challenge mode
- description: Request the game to change the sell setting in a marketplace. Marketplace must not be configured in the same tick, as it was placed or removed.
- content: 
```json
{
    "coord": CubeCoordinate,
    "ratio": {
        // sell percentage, between 0 and 1
        "food": float,
        "materials": float,
    },
}
```

#### Action: `ConfigureRgbTile`
- gamemode: creative
- description: Request the game to change the color value of the RGB tile.
- content: 
```json
{
    "coord": CubeCoordinate,
    "rgb": {
        // color values [0 - 255]
        "r": int,
        "g": int,
        "b": int,
    },
}
```

#### Action: `ConfigureChameleonTile`
- gamemode: creative
- description: Request the game to change the form/model and text of a chameleon tile.
- content: 
```json
{
    "coord": CubeCoordinate,
    "data": {
        "model": TileType,
        "text": string,
    },
}
```

#### Action: `GetTileStatus`
- gamemode: creative
- description: Requests the game to send a `Manual` [TileStatusEvent](#event-tilestatus).
- content: `CubeCoordinate`

#### Action `CreativeSettings`
- gamemode: creative
- description: Enables / Disables diverse user functions in creative mode. Can be usefull for custom game modes.
- content: 
```json
{
    "move_cam": bool,      // allow user to change cam center point
    "rotate_cam": bool,    // allow user to rotate cam
    "zoom_cam": bool,      // allow user to zoom cam
    "tile_popup": bool,    // allow user to open tile info / settings popup onclick
    "place_tiles": bool,   // allow user to place tiles
    "remove_tiles": bool,  // allow user to remove tiles
    "change_inv": bool,    // allow user to change inventory content
    "show_inv_slots": bool // show inventory slots
}
```

#### Action `SetCam`
- gamemode: creative
- description: Sets the position of the cam.
- content: 
```json
{
    "focus": { "x": float, "y": float, "z": float },
    "orbit": { "x": float, "y": float },
    "radius": float,
}
```

#### Action `SetInv`
- gamemode: creative
- description: Sets the position of the cam. Empty slots have to be filled with null or they will disappear. The maximum is 5.
- content: 
```json
[
    TileType | null,
    /* ... */
]
```

### Datatypes
### Tile
Tile, with optional data. If a value is null, the key may be skipped.
```json
{
    "coord": CubeCoordinate,
    "tile": TileType, 
    "data": null | { "type": string, /* data */ },
    "reclaim": null | float // remaining time to reclaim a tile
}
```

#### CubeCoordinate
e.g.:
```json
{
    "q": int,
    "r": int,
    "s": int,
}
```

#### TileType
A string equal to the name of the Tile.
To get a complete list, look at the enum `TileType`
in the [tiles.rs](../../src/game/tiles/mod.rs) file.

#### RewardType
A string equal to the name of the Reward.
To get a complete list, look at the enum `RewardType`
in the [rewards.rs](../../src/game/rewards.rs) file.

#### Metrics
e.g.:
```json
{
    "food": float,
    "materials": float,
    "money": float,
}
```

#### BotEventKind
A string equal to the name of the EventKind.
To get a complete list, look at the enum `BotEventKind`
in the [bot.rs](../../src/game/bot/mod.rs) file.
(The enum is generated by a macro, but the names are still readable.)
