#import bevy_pbr::{
    mesh_view_bindings::globals, 
    pbr_fragment::pbr_input_from_standard_material,
    pbr_functions::alpha_discard,
}

#ifdef PREPASS_PIPELINE
#import bevy_pbr::{
    prepass_io::{VertexOutput, FragmentOutput},
    pbr_deferred_functions::deferred_output,
}
#else
#import bevy_pbr::{
    forward_io::{VertexOutput, FragmentOutput},
    pbr_functions::{apply_pbr_lighting, main_pass_post_lighting_processing},
}
#endif

// Oklab license:
// Copyright (c) 2020 Björn Ottosson
// Distributed under an MIT license: https://bottosson.github.io/misc/License.txt

const one_third = 1.0 / 3.0;

fn linear_srgb_to_oklab(c: vec3<f32>) -> vec3<f32> {
    let l = 0.4122214708f * c.r + 0.5363325363f * c.g + 0.0514459929f * c.b;
	let m = 0.2119034982f * c.r + 0.6806995451f * c.g + 0.1073969566f * c.b;
	let s = 0.0883024619f * c.r + 0.2817188376f * c.g + 0.6299787005f * c.b;

    let l_ = pow(l, one_third);
    let m_ = pow(m, one_third);
    let s_ = pow(s, one_third);

    return vec3<f32>(
        0.2104542553*l_ + 0.7936177850*m_ - 0.0040720468*s_,
        1.9779984951*l_ - 2.4285922050*m_ + 0.4505937099*s_,
        0.0259040371*l_ + 0.7827717662*m_ - 0.8086757660*s_,
    );
}

fn oklab_to_linear_srgb(c: vec3<f32>) -> vec3<f32> {
    let L = c.r;
    let a = c.g;
    let b = c.b;

    let l_ = L + 0.3963377774 * a + 0.2158037573 * b;
    let m_ = L - 0.1055613458 * a - 0.0638541728 * b;
    let s_ = L - 0.0894841775 * a - 1.2914855480 * b;

    let l = l_ * l_ * l_;
    let m = m_ * m_ * m_;
    let s = s_ * s_ * s_;

    return vec3<f32>(
         4.0767416621 * l - 3.3077115913 * m + 0.2309699292 * s,
        -1.2684380046 * l + 2.6097574011 * m - 0.3413193965 * s,
        -0.0041960863 * l - 0.7034186147 * m + 1.7076147010 * s,
    );
}

fn ok_desaturate(c: vec4<f32>) -> vec4<f32> {
    let lab = linear_srgb_to_oklab(c.rgb);
    let desat = vec3<f32>(0.6, 0.0, 0.0); 
    return vec4<f32>(oklab_to_linear_srgb(lab * desat), c.a);
}


// hack to get var 4 Byte alligned: start, duration, height
@group(2) @binding(100)
var<uniform> data: vec4<f32>;

@fragment
fn fragment(
    in: VertexOutput,
    @builtin(front_facing) is_front: bool,
) -> FragmentOutput {
    let start = data.x;
    let duration = data.y;
    let height = data.z;
    let base_height = data.w;

    // generate a PbrInput struct from the StandardMaterial bindings
    var pbr_input = pbr_input_from_standard_material(in, is_front);

    // we can optionally modify the input before lighting and alpha_discard is applied
    var color = pbr_input.material.base_color;

    let y = in.world_position.y - base_height;

    let desat = ok_desaturate(color);

    let now = globals.time;

    let time_diff = now - start;

    let progress = time_diff + step(time_diff, 0.0) * 3600.0;

    let threshold = mix(0.2005, height, progress / duration);

    let mask = smoothstep(threshold+0.01, threshold-0.01, y);

    pbr_input.material.base_color = mix(desat, color, mask);

    // alpha discard
    pbr_input.material.base_color = alpha_discard(pbr_input.material, pbr_input.material.base_color);

#ifdef PREPASS_PIPELINE
    // in deferred mode we can't modify anything after that, as lighting is run in a separate fullscreen shader.
    let out = deferred_output(in, pbr_input);
#else
    var out: FragmentOutput;
    // apply lighting
    out.color = apply_pbr_lighting(pbr_input);

    out.color = main_pass_post_lighting_processing(pbr_input, out.color);

#endif

    return out;
}
