use std::{
    iter::repeat_with,
    net::{TcpListener, TcpStream},
    thread,
    time::Duration,
};

use http::Uri;
use terratactician_expandoria::websocket::{WebSocket, WebSocketConnector};
use tungstenite::accept;

const AUTHORITY: &str = "127.0.0.1:9000";

#[test]
fn test_websocket() {
    thread::spawn(echo_server);
    thread::sleep(Duration::from_millis(1000));
    let mut connector = WebSocketConnector::new(
        Uri::builder()
            .scheme("ws")
            .authority(AUTHORITY)
            .path_and_query("/")
            .build()
            .unwrap(),
    )
    .expect("WebSocket failed to create a connector");
    let mut ws: WebSocket = loop {
        match connector
            .step_connect()
            .expect("WebSocket failed to connect to server.")
        {
            Some(ws) => break ws,
            None => (),
        };
        thread::sleep(Duration::from_millis(10));
    };

    let mut test = String::default();
    for _ in 0..10 {
        test += &repeat_with(fastrand::alphanumeric)
            .take(10000)
            .collect::<String>();

        ws.write(test.clone())
            .expect("WebSocket client: write failed.");

        // sleep to simplify test
        thread::sleep(Duration::from_millis(50));
        assert_eq!(
            ws.read::<String>()
                .expect("WebSocket client: read failed.")
                .expect("No message returned"),
            test,
            "WebSocket response must be equal to send data."
        );
    }
}

fn echo_server() {
    let server = TcpListener::bind(AUTHORITY).unwrap();
    for stream in server.incoming() {
        match stream {
            Ok(stream) => thread::spawn(move || handle_client(stream)),
            Err(e) => panic!("Error accepting tcp client: {}", e),
        };
    }
}

fn handle_client(stream: TcpStream) {
    let mut ws = accept(stream).expect("tcp stream should be accepted by tungstenite.");
    loop {
        match ws
            .read()
            .expect("WebSocket server must not throw an error.")
        {
            tungstenite::Message::Text(txt) => ws
                .send(tungstenite::Message::Text(txt))
                .expect("WebSocket server must be able to send."),
            _ => {}
        };
    }
}
