use bevy::{ecs::system::RunSystemOnce, prelude::State};
use terratactician_expandoria::{
    coordinates::{CubeCoordinate, OffsetCoordinate},
    game::{
        gamemodes::challenge::{init_challenge_mode, ChallengeModeConfig},
        map::TileSaveData,
        metrics::MetricsRate,
        resources::{forest, wheat, TilePlacedEvent},
        tiles::{AddTile, TileType},
    },
    AppBuilder, AppState, DisplayMode,
};

fn started_game(config: ChallengeModeConfig, register_logger: bool) -> bevy::app::App {
    let mut app = AppBuilder {
        display_mode: DisplayMode::Headless,
        #[cfg(feature = "graphics")]
        enable_config_io: false,
        register_logger,
    }
    .build();
    app.finish();
    app.cleanup();
    // this loop is optional. This tests do not need to wait for assets.
    loop {
        if let AppState::AppMenu = app.world_mut().resource::<State<AppState>>().get() {
            break;
        }
        if let Some(state) = app.should_exit() {
            panic!("Game exited premature: {state:?}");
        }
        app.update();
    }
    app.update();
    *app.world_mut().resource_mut::<ChallengeModeConfig>() = config;
    app.world_mut()
        .run_system_once(init_challenge_mode)
        .unwrap();
    app.update();
    app
}

#[test]
fn test_forrest() {
    let mut app = started_game(ChallengeModeConfig::default(), true);
    let mut commands = app.world_mut().commands();
    const FOREST_AMOUNT: isize = 5;
    for i in 0..FOREST_AMOUNT {
        commands
            .spawn_empty()
            .queue(AddTile::new(TileType::Forest, OffsetCoordinate::new(0, i)).without_scene());
    }
    app.world_mut()
        .send_event(TilePlacedEvent(TileSaveData::new(
            CubeCoordinate::default(),
            TileType::Forest,
        )));
    app.update();
    app.update();
    let rate = app.world().resource::<MetricsRate>();
    assert_eq!(
        forest::BASE_PRODUCTION_RATE * FOREST_AMOUNT as f32
            + (forest::BONUS_PER_NEIGHBOR * 2.0) * (FOREST_AMOUNT as f32 - 1.0),
        rate.materials
    );
}

#[test]
fn test_wheat() {
    let mut app = started_game(ChallengeModeConfig::default(), false);
    const WHEAT_AMOUNT: isize = wheat::MAXIMUM_GROUP_SIZE as isize + 1;
    let mut last = 0.0;
    for i in 0..WHEAT_AMOUNT {
        app.world_mut()
            .commands()
            .spawn_empty()
            .queue(AddTile::new(TileType::Wheat, OffsetCoordinate::new(0, i)).without_scene());
        app.world_mut()
            .send_event(TilePlacedEvent(TileSaveData::new(
                CubeCoordinate::default(),
                TileType::Forest,
            )));
        app.update();
        app.update();
        let rate = app.world().resource::<MetricsRate>();
        assert!(
            (last <= rate.food - wheat::BASE_PRODUCTION_RATE)
                ^ (i >= wheat::OPTIMAL_GROUP_SIZE as isize),
            "{}",
            format!("amount: {i}, last: {last}, current: {}", rate.food)
        );
        last = rate.food;
    }
}
