use std::time::Duration;

use bevy::{ecs::system::RunSystemOnce, time::Time};
use terratactician_expandoria::{
    coordinates::{CubeCoordinate, OffsetCoordinate},
    game::{
        gamemodes::challenge::{init_challenge_mode, ChallengeModeConfig},
        map::TileSaveData,
        metrics::{Metrics, MetricsRate},
        resources::TilePlacedEvent,
        round::{METRICS_BOX_TRACKING_SIZE, ROUND_DURATION},
        tiles::{AddTile, TileType},
        time::{Game, StepEvent},
    },
    AppBuilder, DisplayMode,
};

fn started_game(config: ChallengeModeConfig) -> bevy::app::App {
    let mut app = AppBuilder {
        display_mode: DisplayMode::Headless,
        #[cfg(feature = "graphics")]
        enable_config_io: false,
        register_logger: true,
    }
    .build();
    app.finish();
    app.cleanup();
    app.update();
    *app.world_mut().resource_mut::<ChallengeModeConfig>() = config;
    app.world_mut()
        .run_system_once(init_challenge_mode)
        .unwrap();
    app.update();
    app
}

#[test]
fn basic_time_step() {
    let mut app = started_game(ChallengeModeConfig::default());
    let mut commands = app.world_mut().commands();
    const FOREST_AMOUNT: isize = 5;
    for i in 0..FOREST_AMOUNT {
        commands
            .spawn_empty()
            .queue(AddTile::new(TileType::Forest, OffsetCoordinate::new(0, i)).without_scene());
    }
    app.world_mut()
        .send_event(TilePlacedEvent(TileSaveData::new(
            CubeCoordinate::default(),
            TileType::Forest,
        )));
    app.update();
    app.update();
    let rate = app.world().resource::<MetricsRate>().clone();
    let x = METRICS_BOX_TRACKING_SIZE * ROUND_DURATION.as_secs() as usize;
    // Simulate for x ticks.
    // After ROUND_DURATION ticks a phase change will occur, but due to headless mode, it will continue to run.
    // After x ticks the game is lost. (stopping test)
    for i in 0..x {
        println!(
            "virtual time: {}",
            app.world_mut().resource::<Time<Game>>().elapsed_secs_f64()
        );
        let metrics = app.world().resource::<Metrics>();
        assert_eq!(rate.materials * i as f32, metrics.materials);
        app.world_mut()
            .send_event(StepEvent(Duration::from_secs_f64(1.0)));

        // three updates are needed, to process events and state updates.
        app.update();
        app.update();
        app.update();
    }
}
