"use strict";

const WS_NOT_EXISTING = -1;
const WS_CONNECTING = 0;
const WS_CONNECTED = 1;
const WS_CLOSED = 2;
const WS_URL = 3;
const WS_FATAL = 4;

/**
 * Represents a WebSocket connection and its state.
 * @typedef {Object} WebSocketConnection
 * @property {WebSocket} socket - The WebSocket instance.
 * @property {number} state - The state of the WebSocket connection. Possible values:
 *   - 0: Connection is in the CONNECTING state.
 *   - 1: Connection is in the OPEN state.
 *   - 2: Connection is in the CLOSING state.
 *   - 3: Connection is in the CLOSED state.
 * @property {Array} received - An array to store received data or messages.
 */

/**
 * An instance of WebSocketConnection.
 * @type {{[key: number]: WebSocketConnection}}
 */
let ws = {};
let i = 0;

/**
 * Create a WebSocket with id.
 * @param {String} url 
 * @param {Number} id 
 * @returns ws id
 */
function ws_open_js(url, id) {
    let conn;
    try {
        conn = {
            socket: new WebSocket(url),
            url,
            state: WS_CONNECTING,
            received: []
        };
    } catch (error) {
        return -1;
    }

    conn.socket.onopen = (s, e) => {
        conn.state = WS_CONNECTED;
    };
    conn.socket.onmessage = (e) => {
        conn.received.push(e.data.toString());
    };
    conn.socket.onerror = (e) => {
        conn.state = WS_URL;
    };
    conn.socket.onclose = (e) => {
        if (!e.wasClean) {
            if (conn.state != WS_URL) conn.state = WS_FATAL;
        } else {
            conn.state = WS_CLOSED;
        }

    };
    ws[id] = conn;
    return id;
}

/**
 * Creates new websocket with url.
 * @param {String} url 
 * @returns 
 */
function ws_open(url) {
    return ws_open_js(url, i++);
}

function ws_write(id, txt) {
    if (ws[id] != null && ws[id].socket != null && ws[id].socket.readyState == WebSocket.OPEN) {
        ws[id].socket.send(txt);
        return true;
    }
    return false;
}
function ws_read(id) {
    return ws[id].received.shift();
}
function ws_available(id) {
    if (ws[id] == null || !Array.isArray(ws[id].received)) return -1;
    return ws[id].received.length;
}

function ws_state(id) {
    return ws[id]?.state ?? WS_NOT_EXISTING;
}

function ws_close(id) {
    if (ws[id] == null || ws[id].socket == null) return;
    ws[id].socket.close();
    ws[id] = { state: ws[id].state > 1 ? ws[id].state : WS_CLOSED };
}

window.ws_open = ws_open;
window.ws_write = ws_write;
window.ws_read = ws_read;
window.ws_state = ws_state;
window.ws_available = ws_available;
window.ws_close = ws_close;

export {
    ws_open,
    ws_write,
    ws_read,
    ws_state,
    ws_available,
    ws_close,
};