use crate::{
    game::{
        bot::{
            challenge_bot::start_challenge_bot_game, creative_bot::start_creative_bot_game,
            BotConnectingState,
        },
        gamemodes::{
            campaign::{
                levels::{init_campaign_mode, CampaignGroupWrapper},
                CampaignModeConfig, LEVEL_TRANSLATION_ID_BASE,
            },
            challenge::{init_challenge_mode, ChallengeModeConfig},
            creative::{init_creative_mode, CreativeModeConfig},
            zen::{init_zen_mode, ZenModeConfig},
            GameMode,
        },
    },
    i18n::Localization,
    prelude::*,
    settings::{ActiveSettingsBank, PersistentSettingsBank},
    AppState, GameConfig,
};
use bevy::{ecs::system::RunSystemOnce, prelude::*};
use bevy_egui::{
    egui::{self, Align2, FontId, RichText},
    EguiContexts,
};

use super::{
    theme::{self, *},
    UiClickEvent,
};

#[derive(Resource, Default)]
pub struct SetupMenuPlugin;
impl Plugin for SetupMenuPlugin {
    fn build(&self, app: &mut App) {
        app.add_systems(
            Update,
            (ui_setup_menu_system).run_if(in_state(AppState::SetupMenu)),
        );
        app.add_systems(
            OnEnter(AppState::SetupMenu),
            (generate_new_seed, refill_memory).chain(),
        );
    }
}

/// generates a new random seed
pub fn generate_new_seed(
    config: ResMut<GameConfig>,
    mut challenge_config: ResMut<ChallengeModeConfig>,
    mut zen_config: ResMut<ZenModeConfig>,
) {
    if !config.skip_seed {
        challenge_config.seed = fastrand::u64(..).to_string();
        zen_config.seed.clone_from(&challenge_config.seed);
    }
}

/// prefills the `GameConfigMemory` with data from the game_behaviour settings
pub fn refill_memory(
    mut config: ResMut<GameConfig>,
    mut challenge_config: ResMut<ChallengeModeConfig>,
    mut campaign_config: ResMut<CampaignModeConfig>,
    settings: Res<ActiveSettingsBank>,
) {
    if !config.skip_mode {
        challenge_config.difficulty = settings.game_behaviour.difficulty.clone();
        campaign_config.level = settings.game_behaviour.campaign_level.clone();
        config.selected_gamemode = settings.game_behaviour.gamemode.clone();
    }
    if !config.skip_bot {
        config.bot_url.clone_from(&settings.game_behaviour.bot_url);
        config.use_bot = settings.game_behaviour.bot_usage;
    }
    if !config.skip_recorder {
        config
            .replay_name
            .clone_from(&settings.game_behaviour.replay_name);
        config.activate_recorder = settings.game_behaviour.activate_recorder;
    }
}

#[allow(clippy::too_many_arguments)]
pub fn ui_setup_menu_system(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut app_state: ResMut<NextState<AppState>>,
    mut settings: ResMut<ActiveSettingsBank>,
    mut settings_bank_a: ResMut<PersistentSettingsBank>,
    mut config: ResMut<GameConfig>,
    mut challenge_config: ResMut<ChallengeModeConfig>,
    mut zen_config: ResMut<ZenModeConfig>,
    mut campaign_config: ResMut<CampaignModeConfig>,
    mut _creative_config: ResMut<CreativeModeConfig>,
    localization: Res<Localization>,
    mut click_event: EventWriter<UiClickEvent>,
    mut commands: Commands,
    mut bot_state: ResMut<BotConnectingState>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let my_frame = egui::Frame {
        fill: BACKGROUND_COLOR,
        inner_margin: egui::Margin::same((window.y * MARGIN_SIZE) as i8),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };

    egui::Area::new("setup_settings_area".into())
        .anchor(Align2::CENTER_CENTER, [0., 0.])
        .show(egui, |ui| {
            ui.style_mut().interaction.selectable_labels = false;

            ui.label(
                RichText::new(localization.translate("game-setup-Game_Setup"))
                    .font(FontId::new(
                        window.y * BIG_TEXT_SIZE,
                        egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                    ))
                    .color(TITLE_TEXT_COLOR),
            );
            ui.add_space(window.y * SPACE_SIZE);

            if *bot_state == BotConnectingState::None {
                // setting up
                ui.columns(2, |columns| {
                    {
                        let ui = &mut columns[0];
                        ui.label(
                            RichText::new(localization.translate("gamemode"))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                ))
                                .color(NORMAL_TEXT_COLOR),
                        );
                    };
                    {
                        let ui = &mut columns[1];
                        egui::ComboBox::from_id_salt("gamemode")
                            .selected_text(
                                localization
                                    .translate(config.selected_gamemode.get_translation_id()),
                            )
                            .show_ui(ui, |ui| {
                                for gm in GameMode::list() {
                                    if ui
                                        .selectable_value(
                                            &mut config.selected_gamemode,
                                            gm.clone(),
                                            localization.translate(gm.get_translation_id()),
                                        )
                                        .clicked()
                                    {
                                        settings.game_behaviour.gamemode =
                                            config.selected_gamemode.clone();
                                        settings_bank_a.game_behaviour.gamemode =
                                            settings.game_behaviour.gamemode.clone();
                                        click_event.send(UiClickEvent);
                                    };
                                }
                            });
                    };
                });

                match config.selected_gamemode {
                    GameMode::Challenge => {
                        ui.label(
                            RichText::new(localization.translate("setup-menu-seed"))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                ))
                                .color(NORMAL_TEXT_COLOR),
                        );

                        ui.add(
                            egui::text_edit::TextEdit::singleline(&mut challenge_config.seed)
                                .min_size(egui::Vec2::new(window.x * 0.5, window.y * 0.02))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                )),
                        );
                        ui.add_space(window.y * SPACE_SIZE);

                        /* ui.columns(2, |columns| {
                            columns[0].label(
                                RichText::new(localization.translate("setup-menu-difficulty"))
                                    .font(FontId::new(
                                        window.y * SMALL_TEXT_SIZE,
                                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                    ))
                                    .color(NORMAL_TEXT_COLOR),
                            );
                            egui::ComboBox::from_id_source("difficulty")
                                .selected_text(challenge_config.difficulty.get_name(&localization))
                                .show_ui(&mut columns[1], |ui| {
                                    for dif in Difficulty::list() {
                                        let difficulty_name = dif.get_name(&localization);
                                        if ui
                                            .selectable_value(
                                                &mut challenge_config.difficulty,
                                                dif,
                                                difficulty_name,
                                            )
                                            .clicked()
                                        {
                                            settings.game_behaviour.difficulty =
                                                challenge_config.difficulty.clone();
                                            settings_bank_a.game_behaviour.difficulty =
                                                settings.game_behaviour.difficulty.clone();
                                            click_event.send(UiClickEvent);
                                        };
                                    }
                                });
                        }); */
                    }
                    GameMode::Creative => (),
                    GameMode::Campaign => {
                         ui.columns(2, |columns| {
                             columns[0].label(
                                 RichText::new(localization.translate("setup-menu-group"))
                                     .font(FontId::new(
                                         window.y * SMALL_TEXT_SIZE,
                                         egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                     ))
                                     .color(NORMAL_TEXT_COLOR),
                             );
                            egui::ComboBox::from_id_salt("group")
                                .selected_text(localization.translate(&format!("{}-{}", LEVEL_TRANSLATION_ID_BASE, campaign_config.level.get_group_id())))
                                .show_ui(&mut columns[1], |ui| {
                                    for group in CampaignGroupWrapper::list_ids() {
                                        if ui
                                            .selectable_value(
                                                &mut campaign_config.level,
                                                CampaignGroupWrapper::from_group_default(group).expect("The group id was taken from the group id list, so it has to exist"),
                                                localization.translate(&format!("{}-{}", LEVEL_TRANSLATION_ID_BASE, group)),
                                            )
                                            .clicked()
                                        {
                                            // backup to settings
                                        settings.game_behaviour.campaign_level =
                                                campaign_config.level.clone();
                                        settings_bank_a.game_behaviour.campaign_level =
                                                campaign_config.level.clone();
                                            click_event.send(UiClickEvent);
                                        };
                                    }
                                });
                        });
                         ui.columns(2, |columns| {
                            columns[0].label(
                                RichText::new(localization.translate("setup-menu-level"))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                ))
                                .color(NORMAL_TEXT_COLOR),
                            );
                            let group = campaign_config.level.get_group_id();
                            egui::ComboBox::from_id_salt("level")
                                .selected_text(localization.translate(&format!("{}-{}-{}", LEVEL_TRANSLATION_ID_BASE, group, campaign_config.level.get_level_id())))
                                .show_ui(&mut columns[1], |ui| {
                                    for level in CampaignGroupWrapper::list_level_ids(group).expect("The group id was taken from a valid CampaignGroupWrapper, so the group has to exist") {
                                        if ui
                                            .selectable_value(
                                                &mut campaign_config.level,
                                                CampaignGroupWrapper::from_ids(group, level).expect("The level was taken from the groups level list, so it has to exist"),
                                                localization.translate(&format!("{}-{}-{}", LEVEL_TRANSLATION_ID_BASE, group, level)),
                                            )
                                            .clicked()
                                        {
                                            // backup to settings
                                        settings.game_behaviour.campaign_level =
                                                campaign_config.level.clone();
                                        settings_bank_a.game_behaviour.campaign_level =
                                                campaign_config.level.clone();
                                            click_event.send(UiClickEvent);
                                        };
                                    }
                                });
                        });
                    },
                    GameMode::Zen => {
                        ui.label(
                            RichText::new(localization.translate("setup-menu-seed"))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                ))
                                .color(NORMAL_TEXT_COLOR),
                        );

                        ui.add(
                            egui::text_edit::TextEdit::singleline(&mut zen_config.seed)
                                .min_size(egui::Vec2::new(window.x * 0.5, window.y * 0.02))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                )),
                        );
                        ui.add_space(window.y * SPACE_SIZE);
                    }
                }

                ui.add_space(window.y * SPACE_SIZE * 2.0);

                let allow_bot_connections = match config.selected_gamemode {
                    // allow connectign bot in creative and challenge mode
                    GameMode::Challenge | GameMode::Creative | GameMode::Zen => true,
                    // only allow bot connections in supported campaign levels
                    GameMode::Campaign => campaign_config.level.get_bot_type().is_some()
                };
                if allow_bot_connections {
                    ui.horizontal(|ui| {
                        ui.label(
                            RichText::new(localization.translate("bot-url"))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                ))
                                .color(NORMAL_TEXT_COLOR),
                        );
                        ui.add(
                            egui::text_edit::TextEdit::singleline(&mut config.bot_url)
                                .min_size(egui::Vec2::new(window.x * 0.25, window.y * 0.02))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                )),
                        );
                        ui.checkbox(&mut config.use_bot, localization.translate("bot-use"));
                    });

                    ui.add_space(window.y * SPACE_SIZE * 2.0);
                }

                // game recorder
                if config.selected_gamemode == GameMode::Challenge || config.selected_gamemode == GameMode::Zen {

                    ui.horizontal(|ui| {
                        ui.label(
                        RichText::new(localization.translate("game-record-name"))
                            .font(FontId::new(
                                window.y * SMALL_TEXT_SIZE,
                                egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                            ))
                            .color(NORMAL_TEXT_COLOR),
                        );
                        ui.add(
                            egui::text_edit::TextEdit::singleline(&mut config.replay_name)
                                .min_size(egui::Vec2::new(window.x * 0.2, window.y * 0.02))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                )),
                        );
                        ui.checkbox(&mut config.activate_recorder, localization.translate("activate-recorder"));
                    });
                    ui.add_space(window.y * SPACE_SIZE * 2.0);
                }

                ui.horizontal_centered(|ui| {
                    ui.style_mut().text_styles.insert(
                        egui::TextStyle::Button,
                        FontId::new(
                            window.y * SECONDARY_BUTTON_TEXT_SIZE,
                            egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                        )
                    );
                    if ui
                        .add_sized(
                            (window * theme::SECONDARY_BUTTON_SIZE).cast(),
                            egui::Button::new(
                                RichText::new(localization.translate("back"))
                                    .color(BUTTON_TEXT_COLOR),
                            )
                            .fill(BUTTON_COLOR),
                        )
                        .clicked()
                    {
                        app_state.set(AppState::AppMenu);
                        click_event.send(UiClickEvent);
                    }
                    if ui
                        .add_sized(
                            (window * theme::SECONDARY_BUTTON_SIZE).cast(),
                            egui::Button::new(
                                RichText::new(localization.translate("start-game"))
                                    .color(BUTTON_TEXT_COLOR),
                            )
                            .fill(BUTTON_COLOR),
                        )
                        .clicked()
                    {
                        click_event.send(UiClickEvent);
                        settings.game_behaviour.bot_usage = config.use_bot;
                        settings_bank_a.game_behaviour.bot_usage = config.use_bot;
                        settings.game_behaviour.bot_url.clone_from(&config.bot_url);
                        settings_bank_a
                            .game_behaviour
                            .bot_url
                            .clone_from(&config.bot_url);


                        settings.game_behaviour.activate_recorder = config.activate_recorder;
                        settings_bank_a.game_behaviour.activate_recorder = config.activate_recorder;

                        settings.game_behaviour.replay_name.clone_from( &config.replay_name);
                        settings_bank_a.game_behaviour.replay_name.clone_from( &config.replay_name);

                        let mode = config.selected_gamemode.clone();
                        if config.use_bot && allow_bot_connections {
                            let campaign_level = campaign_config.level.clone();
                            commands.queue(move |w: &mut World| {
                                match mode {
                                    GameMode::Challenge => {
                                        w.run_system_once(start_challenge_bot_game).unwrap();
                                    }
                                    GameMode::Zen => {
                                        w.run_system_once(start_challenge_bot_game).unwrap();
                                    }
                                    GameMode::Creative => {
                                        w.run_system_once(start_creative_bot_game).unwrap();
                                    }
                                    GameMode::Campaign => {
                                        // try to initialize a campaign bot for the given level
                                        if !campaign_level.init_bot(w) {
                                            // the initialization returns false if the level does not support bots
                                            // in this case we have to fall back to the normal campaign init method,
                                            // as we might still have the old bot enabled state for compatible levels
                                            // NOTE: should never happen, as this block only runs when bot connections are allowed
                                            w.run_system_once(init_campaign_mode).unwrap();
                                        }
                                    },
                                };
                            });
                        } else {
                            commands.queue(move |w: &mut World| {
                                match mode {
                                    GameMode::Challenge => w.run_system_once(init_challenge_mode).unwrap(),
                                    GameMode::Creative => w.run_system_once(init_creative_mode).unwrap(),
                                    GameMode::Campaign => w.run_system_once(init_campaign_mode).unwrap(),
                                    GameMode::Zen => w.run_system_once(init_zen_mode).unwrap(),
                                };
                            });
                        }
                    }
                });
            } else {
                // connecting or failed
                ui.label(
                    RichText::new(format!(
                        "{}: {}",
                        localization.translate("bot-state"),
                        localization.translate(bot_state.get_translation_id())
                    ))
                    .font(FontId::new(
                        window.y * SMALL_TEXT_SIZE,
                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                    ))
                    .color(NORMAL_TEXT_COLOR),
                );
                if !bot_state.is_waiting_state()
                    && ui
                        .add_sized(
                            (window * theme::SECONDARY_BUTTON_SIZE).cast(),
                            egui::Button::new(
                                RichText::new(localization.translate("back"))
                                    .color(BUTTON_TEXT_COLOR),
                            )
                            .fill(BUTTON_COLOR),
                        )
                        .clicked()
                {
                    *bot_state = BotConnectingState::None;
                }
            }
        });

    // center panel
    egui::CentralPanel::default()
        .frame(my_frame)
        .show(egui, |_ui| {});
}
