use crate::{
    i18n::Localization,
    prelude::*,
    settings::{keybindings::KeybindingSettings, ActiveSettingsBank},
    ui::{theme::TITLE_TEXT_COLOR, UiClickEvent},
};
use bevy::{input::keyboard::KeyboardInput, prelude::*};

use super::{
    KeyRemap, NORMAL_TEXT_COLOR, PARAGRAPH_FONT_NAME, POPUP_COLOR_BG, POPUP_RADIUS,
    POPUP_STROKE_COLOR, SMALL_HEADING_SIZE, SMALL_TEXT_SIZE,
};
use bevy_egui::{
    egui::{self, emath, Align2, FontId, RichText},
    EguiContexts,
};

#[derive(Resource, Default)]
pub struct SetKeybindLogic {
    pub show_dialog: bool,
    pub changes: (String, isize),
}

pub fn keybindings_tab(
    ui: &mut egui::Ui,
    window: Vec2,
    mut settings: ResMut<ActiveSettingsBank>,
    mut set_key_bind_logic: ResMut<SetKeybindLogic>,
    localization: &Localization,
    mut click_event: EventWriter<UiClickEvent>,
    mut key_remap_state: ResMut<NextState<KeyRemap>>,
) {
    egui::ScrollArea::vertical()
        .scroll_bar_visibility(egui::scroll_area::ScrollBarVisibility::VisibleWhenNeeded)
        .auto_shrink(false)
        .show(ui, |ui| {
            ui.label(
                RichText::new(localization.translate("mouse-settings"))
                    .font(FontId::new(
                        window.y * SMALL_HEADING_SIZE,
                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                    ))
                    .color(TITLE_TEXT_COLOR),
            );

            egui::Grid::new("mouse_settings")
                .num_columns(2)
                .spacing([40.0, 4.0])
                .striped(true)
                .show(ui, |ui| {
                    ui.add(egui::Label::new(
                        RichText::new(localization.translate("orbit"))
                            .font(FontId::new(
                                window.y * SMALL_TEXT_SIZE,
                                egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                            ))
                            .color(NORMAL_TEXT_COLOR),
                    ));

                    egui::ComboBox::new("Orbit Setting", "")
                        .selected_text(
                            localization.translate(settings.mouse.orbit.get_translation_id()),
                        )
                        .show_ui(ui, |ui| {
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.orbit,
                                    MouseButton::Left,
                                    localization.translate("left-click"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.orbit,
                                    MouseButton::Right,
                                    localization.translate("right-click"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.orbit,
                                    MouseButton::Middle,
                                    localization.translate("middle-mouse-button"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                        });
                    ui.end_row();

                    ui.add(egui::Label::new(
                        RichText::new(localization.translate("pan"))
                            .font(FontId::new(
                                window.y * SMALL_TEXT_SIZE,
                                egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                            ))
                            .color(NORMAL_TEXT_COLOR),
                    ));
                    egui::ComboBox::new("Pan Setting", "")
                        .selected_text(
                            localization.translate(settings.mouse.pan.get_translation_id()),
                        )
                        .show_ui(ui, |ui| {
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.pan,
                                    MouseButton::Left,
                                    localization.translate("left-click"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.pan,
                                    MouseButton::Right,
                                    localization.translate("right-click"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.pan,
                                    MouseButton::Middle,
                                    localization.translate("middle-mouse-button"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                        });
                    ui.end_row();

                    ui.add(egui::Label::new(
                        RichText::new(localization.translate("place"))
                            .font(FontId::new(
                                window.y * SMALL_TEXT_SIZE,
                                egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                            ))
                            .color(NORMAL_TEXT_COLOR),
                    ));
                    egui::ComboBox::new("Place Setting", "")
                        .selected_text(
                            localization.translate(settings.mouse.place.get_translation_id()),
                        )
                        .show_ui(ui, |ui| {
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.place,
                                    MouseButton::Left,
                                    localization.translate("left-click"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.place,
                                    MouseButton::Right,
                                    localization.translate("right-click"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.place,
                                    MouseButton::Middle,
                                    localization.translate("middle-mouse-button"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                        });
                    ui.end_row();

                    ui.add(egui::Label::new(
                        RichText::new(localization.translate("destroy"))
                            .font(FontId::new(
                                window.y * SMALL_TEXT_SIZE,
                                egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                            ))
                            .color(NORMAL_TEXT_COLOR),
                    ));

                    egui::ComboBox::new("Destroy Setting", "")
                        .selected_text(
                            localization.translate(settings.mouse.destroy.get_translation_id()),
                        )
                        .show_ui(ui, |ui| {
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.destroy,
                                    MouseButton::Left,
                                    localization.translate("left-click"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.destroy,
                                    MouseButton::Right,
                                    localization.translate("right-click"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                            if ui
                                .selectable_value(
                                    &mut settings.mouse.destroy,
                                    MouseButton::Middle,
                                    localization.translate("middle-mouse-button"),
                                )
                                .clicked()
                            {
                                click_event.send(UiClickEvent);
                            }
                        });
                    ui.end_row();
                });

            ui.separator();
            ui.label(
                RichText::new(localization.translate("keyboard-settings"))
                    .font(FontId::new(
                        window.y * SMALL_HEADING_SIZE,
                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                    ))
                    .color(TITLE_TEXT_COLOR),
            );

            // make buttons fill grid-cell (more)
            let min_button_size = emath::Vec2::new(window.x / 10.0, window.y / 70.0);
            egui::Grid::new("keyboard_settings")
                .num_columns(4)
                // .min_col_width(window.x/5.0)
                .spacing([window.x / 10.0, window.y / 75.0])
                .striped(true)
                .show(ui, |ui| {
                    for tid in settings.keybindings.list() {
                        ui.label(
                            RichText::new(localization.translate(&tid))
                                .font(FontId::new(
                                    window.y * SMALL_TEXT_SIZE,
                                    egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                                ))
                                .color(NORMAL_TEXT_COLOR),
                        );

                        if let Some(r) = settings.keybindings.get(&tid) {
                            if ui
                                .add(
                                    egui::Button::new(
                                        localization.translate(r.slot1.get_translation_id()),
                                    )
                                    .min_size(min_button_size),
                                )
                                .clicked()
                            {
                                info!(
                                    "Switched to state KeyRemap::Listening | Changes: {:?}",
                                    (tid.to_string(), 0)
                                );
                                key_remap_state.set(KeyRemap::Listening);
                                set_key_bind_logic.show_dialog = true;
                                set_key_bind_logic.changes = (tid.to_string(), 0);
                                click_event.send(UiClickEvent);
                            }
                            if ui
                                .add(
                                    egui::Button::new(
                                        localization.translate(r.slot2.get_translation_id()),
                                    )
                                    .min_size(min_button_size),
                                )
                                .clicked()
                            {
                                info!(
                                    "Switched to state KeyRemap::Listening | Changes: {:?}",
                                    (tid.to_string(), 1)
                                );
                                key_remap_state.set(KeyRemap::Listening);
                                set_key_bind_logic.show_dialog = true;
                                set_key_bind_logic.changes = (tid.to_string(), 1);
                                click_event.send(UiClickEvent);
                            }
                        }
                        if let Some(&mut ref mut _r) = settings.keybindings.get_mut(&tid) {
                            if ui
                                .add(
                                    egui::Button::new(localization.translate("reset"))
                                        .min_size(min_button_size),
                                )
                                .clicked()
                            {
                                *_r = KeybindingSettings::default()
                                    .get(&tid)
                                    .expect("Macro generated")
                                    .clone();
                                click_event.send(UiClickEvent);
                            }
                        }
                        ui.end_row();
                    }
                });
        });
}

pub fn reset_keyboard_input(mut set_key_bind_logic: ResMut<SetKeybindLogic>) {
    *set_key_bind_logic = SetKeybindLogic::default();
}

pub fn keyboard_input(
    mut key_evr: EventReader<KeyboardInput>,
    mut set_key_bind_logic: ResMut<SetKeybindLogic>,
    mut settings: ResMut<ActiveSettingsBank>,
    mut key_remap_state: ResMut<NextState<KeyRemap>>,
) {
    use bevy::input::ButtonState;
    // wait till button pressed
    let mut got_input = false;
    for ev in key_evr.read() {
        if let ButtonState::Released = ev.state {
            if set_key_bind_logic.show_dialog {
                let target = &set_key_bind_logic.changes;
                if let Some(option) = settings.keybindings.get_mut(&target.0) {
                    match ev.key_code {
                        KeyCode::Escape => {
                            info!("Resetting: {} field {}", target.0, target.1);
                            // if ESC => unmap
                            if target.1 == 0 {
                                option.slot1 = None;
                            } else if target.1 == 1 {
                                option.slot2 = None;
                            }
                        }
                        other => {
                            info!("Setting: {} field {} to {:?}", target.0, target.1, other);
                            if target.1 == 0 {
                                option.slot1 = Some(other);
                            } else if target.1 == 1 {
                                option.slot2 = Some(other);
                            }
                        }
                    }
                    got_input = true;
                }
            }
        }
        if got_input {
            // stop listening for new keycodes
            *set_key_bind_logic = SetKeybindLogic::default();
            info!("Switched to state KeyRemap::Deaf (keycode got set or unset)");
            key_remap_state.set(KeyRemap::Deaf);
            break;
        }
    }
}

pub fn key_remap_popup(
    mut contexts: EguiContexts,
    mut key_remap_state: ResMut<NextState<KeyRemap>>,
    mut set_key_bind_logic: ResMut<SetKeybindLogic>,
    localization: Res<Localization>,
) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let my_frame = egui::Frame {
        fill: POPUP_COLOR_BG,
        stroke: egui::Stroke::new(1.5, POPUP_STROKE_COLOR),
        corner_radius: POPUP_RADIUS,
        inner_margin: egui::Margin::same(5),
        outer_margin: egui::Margin::same(0),
        ..Default::default()
    };

    // needed for when window is closed via x
    // thanks to bevy only runs once
    if !set_key_bind_logic.show_dialog {
        // stop listening for KeyRemap
        // reenables things like toggle-fps
        info!("Switched to state KeyRemap::Deaf (dialog got canceled)");
        key_remap_state.set(KeyRemap::Deaf);
    }

    let screen_size = egui.input(|i| i.screen_rect);
    egui::Area::new("key-remap-popup-backdrop".into())
        .order(egui::Order::Background)
        .show(egui, |ui| {
            egui::Frame {
                fill: POPUP_COLOR_BG.gamma_multiply(0.7),
                ..Default::default()
            }
            .show(ui, |ui| {
                if ui
                    .allocate_rect(screen_size, egui::Sense::click())
                    .clicked()
                {
                    set_key_bind_logic.show_dialog = false;
                }
            });
        });

    egui::Window::new(localization.translate(&set_key_bind_logic.changes.0))
        .frame(my_frame)
        .collapsible(false)
        .resizable(false)
        .open(&mut set_key_bind_logic.show_dialog)
        .anchor(Align2::CENTER_CENTER, [0.0, 0.0])
        .order(egui::Order::Foreground)
        .show(egui, |ui| {
            ui.style_mut().interaction.selectable_labels = false;
            ui.label(localization.translate("config-press-any-key"));
            ui.label(localization.translate("config-press-esc-to-unbind"));
        });
}
