use std::collections::BTreeMap;

use about::AboutAppPlugin;
use app_loading::LoadingScreenPlugin;
use bevy::{prelude::*, window::WindowResized};
use bevy_egui::{
    egui::{FontFamily, FontId},
    EguiContexts, EguiPlugin,
};
use commons::CommonsPlugin;
use errors::show_error_screen;
use font_loader::set_fallback_fonts;
use help::HelpAppPlugin;
use in_game::InGameUiPlugin;
use main_menu::MainMenuPlugin;
use menu_bg::MenuBackgroundPlugin;
use notifications::NotificationsPlugin;
use settings_menu::SettingsMenuPlugin;
use setup_menu::SetupMenuPlugin;
use theme::{MONOSPACE_FONT_NAME, PARAGRAPH_FONT_NAME, TITLE_FONT_NAME};

use crate::{game::asset_loading::LACManager, AppState};

use self::font_loader::{AppFonts, FontResourceAsset, FontResourceLoader};

pub mod about;
mod app_loading;
pub mod commons;
pub mod errors;
pub mod font_loader;
pub mod help;
pub mod in_game;
pub mod main_menu;
#[cfg(feature = "material-debugger")]
pub mod material_debugger;
mod menu_bg;
pub mod notifications;
pub mod settings_menu;
pub mod setup_menu;
pub mod theme;

pub const WEBSITE_BASE: &str = "https://terratactician-expandoria.codeberg.page";

/// Can not be used as a run_if system, because some traits are not satisfied.
pub fn is_using_egui(contexts: &mut EguiContexts) -> bool {
    let Some(egui) = contexts.try_ctx_mut() else {
        return true;
    };
    egui.is_using_pointer()
        || egui.is_pointer_over_area()
        || egui.is_context_menu_open()
        || egui.wants_keyboard_input()
}

pub fn is_typing_egui(contexts: &mut EguiContexts) -> bool {
    let Some(egui) = contexts.try_ctx_mut() else {
        return true;
    };
    egui.is_using_pointer() || egui.is_context_menu_open() || egui.wants_keyboard_input()
}

#[derive(Default)]
pub struct UiPlugin;

impl Plugin for UiPlugin {
    fn build(&self, app: &mut bevy::prelude::App) {
        app.register_asset_collection::<AppFonts>();
        app.add_systems(Startup, set_fallback_fonts);

        app.init_asset_loader::<FontResourceLoader>()
            .init_asset::<FontResourceAsset>();

        app.add_event::<UiClickEvent>();

        app.add_plugins(EguiPlugin)
            .add_plugins(MainMenuPlugin)
            .add_plugins(AboutAppPlugin)
            .add_plugins(HelpAppPlugin)
            .add_plugins(SettingsMenuPlugin)
            .add_plugins(SetupMenuPlugin)
            .add_plugins(InGameUiPlugin)
            .add_plugins(CommonsPlugin)
            .add_plugins(LoadingScreenPlugin)
            .add_plugins(MenuBackgroundPlugin)
            .add_plugins(NotificationsPlugin);

        app.add_systems(Update, show_error_screen.run_if(in_state(AppState::Error)));
        app.add_systems(Update, reset_areas_on_resize);

        app.add_systems(
            PreUpdate,
            apply_font_style.run_if(not(in_state(AppState::LoadApp))),
        );

        #[cfg(feature = "material-debugger")]
        {
            use material_debugger::debug_material;
            app.add_systems(Update, debug_material);
        }
    }
}

fn apply_font_style(mut contexts: EguiContexts) {
    use bevy_egui::egui::style::TextStyle::*;
    let text_styles: BTreeMap<_, _> = [
        (
            Heading,
            FontId::new(30.0, FontFamily::Name(TITLE_FONT_NAME.into())),
        ),
        (
            Body,
            FontId::new(18.0, FontFamily::Name(PARAGRAPH_FONT_NAME.into())),
        ),
        (
            Button,
            FontId::new(14.0, FontFamily::Name(PARAGRAPH_FONT_NAME.into())),
        ),
        (
            Small,
            FontId::new(12.0, FontFamily::Name(PARAGRAPH_FONT_NAME.into())),
        ),
        (
            Monospace,
            FontId::new(14.0, FontFamily::Name(MONOSPACE_FONT_NAME.into())),
        ),
    ]
    .into();
    contexts
        .ctx_mut()
        .style_mut(move |style| style.text_styles = text_styles.clone());
}

#[derive(Clone)]
struct WindowSizeData {
    height: f32,
    width: f32,
}

/// number of percent the window size has to change
/// for the egui areas to be reset
/// the lower this value is, the lower the resize changes have to be
const RESIZE_RESET_DISTANCE: f32 = 0.2;

/// resets the area state when the window is resized a given distance
/// prevents weird visual glitches
fn reset_areas_on_resize(
    mut contexts: EguiContexts,
    mut events: EventReader<WindowResized>,
    mut size: Local<Option<WindowSizeData>>,
) {
    if events.is_empty() {
        return;
    }
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };
    for e in events.read() {
        let old_size = (*size).clone();
        *size = Some(WindowSizeData {
            height: e.height,
            width: e.width,
        });
        if let Some(old_size) = old_size {
            let dx = (old_size.width - e.width).abs() / old_size.width;
            let dy = (old_size.height - e.height).abs() / old_size.height;

            if dx <= RESIZE_RESET_DISTANCE && dy <= RESIZE_RESET_DISTANCE {
                // skip reset
                continue;
            }
        }
        // reset if the delta is large enough, or this is the first resize
        egui.memory_mut(|mem| mem.reset_areas());
    }
}

#[derive(Event)]
pub struct UiClickEvent;

#[allow(dead_code)]
#[derive(Debug, Component)]
/// Adds a equi Material Debugger to an Entity. This will do nothing if feature `material-debugger` is disabled.
pub struct DebugMaterial(pub &'static str);
