use crate::{i18n::Localization, prelude::*, AppMenuState, AppState};
use bevy::prelude::*;
use bevy_egui::{
    egui::{self, RichText},
    EguiContexts,
};

use super::{theme::*, UiClickEvent};

#[cfg(not(any(target_arch = "wasm32", target_arch = "wasm64")))]
use bevy::app::AppExit;

pub struct MainMenuPlugin;
impl Plugin for MainMenuPlugin {
    fn build(&self, app: &mut App) {
        app.add_systems(
            Update,
            (ui_main_menu_system).run_if(in_state(AppMenuState::MainMenu)),
        );
    }
}

pub fn ui_main_menu_system(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut app_state: ResMut<NextState<AppState>>,
    mut app_menu_state: ResMut<NextState<AppMenuState>>,
    mut exit: EventWriter<AppExit>,
    localization: Res<Localization>,
    mut click_event: EventWriter<UiClickEvent>,
) {
    let Ok(window) = windows.get_single() else {
        return;
    };

    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let my_frame = egui::Frame {
        fill: BACKGROUND_COLOR.gamma_multiply(0.6),
        inner_margin: egui::Margin::same((window.y * MARGIN_SIZE) as i8),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };

    let title_text = RichText::new("TerraTactician")
        .font(egui::FontId::new(
            window.y * BIG_TEXT_SIZE,
            egui::FontFamily::Name(TITLE_FONT_NAME.into()),
        ))
        .color(TITLE_TEXT_COLOR);
    let subtitle_text = RichText::new("- Expandoria -")
        .font(egui::FontId::new(
            window.y * BIG_TEXT_SIZE / 2.0,
            egui::FontFamily::Name(TITLE_FONT_NAME.into()),
        ))
        .color(TITLE_TEXT_COLOR);

    egui::CentralPanel::default()
        // fill the background with the background color
        .frame(my_frame)
        .show(egui, |_ui| {});

    egui::Area::new("mainmenu-content-area".into())
        .anchor(egui::Align2::CENTER_CENTER, [0.0, 0.0])
        .show(egui, |ui| {
            ui.vertical_centered(|ui| {
                ui.style_mut().wrap_mode = Some(egui::TextWrapMode::Extend);
                ui.style_mut().interaction.selectable_labels = false;
                ui.label(title_text);
                ui.label(subtitle_text);

                ui.style_mut().text_styles.insert(
                    egui::TextStyle::Button,
                    egui::FontId::new(
                        window.y * PRIMARY_BUTTON_TEXT_SIZE,
                        egui::FontFamily::Name(TITLE_FONT_NAME.into()),
                    ),
                );

                ui.add_space(window.y * SPACE_SIZE * 2.0);

                // start game button
                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            RichText::new(localization.translate("start-game"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    app_state.set(AppState::SetupMenu);
                    click_event.send(UiClickEvent);
                }
                ui.add_space(window.y * SPACE_SIZE);

                // settings button
                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            RichText::new(localization.translate("settings"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    app_menu_state.set(AppMenuState::SettingsMenu);
                    click_event.send(UiClickEvent);
                }
                ui.add_space(window.y * SPACE_SIZE);

                // quit button
                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            RichText::new(localization.translate("quit")).color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    click_event.send(UiClickEvent);
                    exit.send(AppExit::Success);
                }
            });
        });
}
