use bevy::{ecs::system::SystemParam, prelude::*, state::state::FreelyMutableState};
use bevy_egui::{
    egui::{self, Align2, FontFamily, FontId, RichText},
    EguiContexts,
};

use crate::{
    game::{
        controls::KeyboardEvent,
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
            },
            challenge::ChallengeGameState,
            zen::ZenGameState,
            GameModeSubStates,
        },
        round::{RoundCounter, RoundSystemSupport},
        GamePauseState,
    },
    i18n::Localization,
    prelude::*,
    settings::ActiveSettingsBank,
    ui::{
        is_using_egui,
        theme::{
            BUTTON_COLOR, BUTTON_TEXT_COLOR, PARAGRAPH_FONT_NAME, PRIMARY_BUTTON_SIZE,
            PRIMARY_BUTTON_TEXT_SIZE, SPACE_SIZE, TITLE_FONT_NAME,
        },
        UiClickEvent,
    },
    AppState,
};

use crate::ui::theme::{
    BETWEENROUNDS_BACKGROUND_COLOR, GAMEOVER_BACKGROUND_COLOR, GRID_TEXT_COLOR,
    OVERLAY_TITLE_TEXT_COLOR,
};

use super::game_export::show_export_window;

macro_rules! define_rounds_ui_plugin {
    ($({
        // gamemode substates
        states: $states:ident,
        // state, in which the `BetweenRounds` screen is shown
        // has to be a substate of states
        between: $between:expr,
        // state, in which the `GameOver` screen is shown
        // has to be a substate of states
        $(end: $end:expr$(,)?)?
    }),*) => {
        pub struct RoundsUiPlugin;
        impl Plugin for RoundsUiPlugin {
            fn build(&self, app: &mut bevy::prelude::App) {
                $(
                    app.add_systems(
                        Update,
                        (
                            show_round_transition_screen::<$states>.before(show_export_window),
                            skip_round_transition::<$states>,
                            keybinding_next_round::<$states>,
                        )
                            .run_if(in_state($between))
                            .run_if(in_state(GamePauseState::Paused)),
                    );
                    $(
                        app.add_systems(
                            Update,
                            show_game_over_screen::<$states>
                                .run_if(in_state($end))
                                .run_if(in_state(GamePauseState::Paused))
                                .before(show_export_window),
                        );
                    )?
                )*
            }
        }
    }
}
define_rounds_ui_plugin![
    {
        states: ChallengeGameState,
        between: ChallengeGameState::BetweenRounds,
        end: ChallengeGameState::GameOver,
    },
    {
        states: CampaignChallengeState,
        between: CampaignChallengeState::BetweenRounds,
        end: CampaignChallengeState::GameOver,
    },
    {
        states: CampaignWettbewerbState,
        between: CampaignWettbewerbState::BetweenRounds,
        end: CampaignWettbewerbState::GameOver,
    },
    {
        states: ZenGameState,
        between: ZenGameState::BetweenRounds,
    }
];

/// Detects mouse and touch input outside of egui elements
/// and continues to the next round if an event was detected
fn skip_round_transition<STATES: GameModeSubStates + FreelyMutableState + RoundSystemSupport>(
    mut contexts: EguiContexts,
    mouse_input: Res<ButtonInput<MouseButton>>,
    touches: Res<Touches>,
    mut next_state: ResMut<NextState<STATES>>,
    mut click_event: EventWriter<UiClickEvent>,
) {
    // don't process events manually if the user interacted with the UI
    if is_using_egui(&mut contexts) {
        return;
    }
    // continue to next round if the player clicks anywhere on screen
    // with the mouse or on a touch screen
    if mouse_input.just_released(MouseButton::Left) || touches.any_just_released() {
        next_state.set(STATES::get_inround_state());
        click_event.send(UiClickEvent);
    }
}

/// shows the between rounds screen
fn show_round_transition_screen<
    STATES: GameModeSubStates + FreelyMutableState + RoundSystemSupport,
>(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut next_state: ResMut<NextState<STATES>>,
    localization: Res<Localization>,
    round_counter: Res<RoundCounter>,
    mut click_event: EventWriter<UiClickEvent>,
) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    // draw background overlay
    let frame = egui::Frame {
        fill: BETWEENROUNDS_BACKGROUND_COLOR.gamma_multiply(0.7),
        inner_margin: egui::Margin::same(12),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };
    egui::CentralPanel::default()
        .frame(frame)
        .show(egui, |_ui| {});

    // render centered widgets
    let window = windows.single();
    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let big_text_size = window.y / 10.0;
    let localization = localization.into_inner();

    egui::Area::new("round-transition-overlay".into())
        .anchor(Align2::CENTER_CENTER, egui::Vec2::new(0.0, 0.0))
        .show(egui, |ui| {
            ui.vertical_centered(|ui| {
                ui.style_mut().interaction.selectable_labels = false;

                ui.label(
                    RichText::new(localization.translate("game-round-end"))
                        .font(FontId::new(
                            big_text_size,
                            FontFamily::Name(TITLE_FONT_NAME.into()),
                        ))
                        .color(OVERLAY_TITLE_TEXT_COLOR),
                );
                ui.label(
                    RichText::new(format!(
                        "{}: {}",
                        localization.translate("game-round-finished"),
                        round_counter.0
                    ))
                    .color(GRID_TEXT_COLOR),
                );

                ui.add_space(SPACE_SIZE * window.y * 3.0);

                ui.style_mut().text_styles.insert(
                    egui::TextStyle::Button,
                    egui::FontId::new(
                        window.y * PRIMARY_BUTTON_TEXT_SIZE / 2.,
                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                    ),
                );

                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            egui::RichText::new(localization.translate("game-round-next"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    next_state.set(STATES::get_inround_state());
                    click_event.send(UiClickEvent);
                }
            });
        });
}

#[derive(SystemParam)]
struct GameOverData<'w> {
    app_state: ResMut<'w, NextState<AppState>>,
    pause_state: ResMut<'w, NextState<GamePauseState>>,
    click_event: EventWriter<'w, UiClickEvent>,
}

/// shows the end of game screen
fn show_game_over_screen<STATES: GameModeSubStates + FreelyMutableState + RoundSystemSupport>(
    windows: Query<&Window>,
    mut contexts: EguiContexts,
    mut game_state: ResMut<NextState<STATES>>,
    localization: Res<Localization>,
    round_counter: Res<RoundCounter>,
    mut ctrl: GameOverData,
) {
    // getting egui_ctx graceful
    let Some(egui) = contexts.try_ctx_mut() else {
        return;
    };

    // draw background overlay
    let frame = egui::Frame {
        fill: GAMEOVER_BACKGROUND_COLOR.gamma_multiply(0.7),
        inner_margin: egui::Margin::same(12),
        outer_margin: egui::Margin::same(0),
        shadow: egui::epaint::Shadow::NONE,
        ..Default::default()
    };
    egui::CentralPanel::default()
        .frame(frame)
        .show(egui, |_ui| {});

    // render centered widgets
    let window = windows.single();
    let window = Vec2::new(window.width(), window.height());

    // don't render UI if window isn't visible
    // when minimizing window size is set to x: 0; y: 0 (only on windows)
    if window.y == 0.0 || window.x == 0.0 {
        return;
    }

    let big_text_size = window.y / 10.0;
    let localization = localization.into_inner();

    egui::Area::new("round-game-over".into())
        .anchor(Align2::CENTER_CENTER, egui::Vec2::new(0.0, 0.0))
        .show(egui, |ui| {
            ui.vertical_centered(|ui| {
                ui.style_mut().interaction.selectable_labels = false;

                ui.label(
                    RichText::new(localization.translate("game-over"))
                        .font(FontId::new(
                            big_text_size,
                            FontFamily::Name(TITLE_FONT_NAME.into()),
                        ))
                        .color(OVERLAY_TITLE_TEXT_COLOR),
                );
                ui.label(
                    RichText::new(format!(
                        "{}: {}",
                        localization.translate("game-over-round"),
                        round_counter.0
                    ))
                    .color(GRID_TEXT_COLOR),
                );

                ui.add_space(SPACE_SIZE * window.y * 3.0);

                ui.style_mut().text_styles.insert(
                    egui::TextStyle::Button,
                    egui::FontId::new(
                        window.y * PRIMARY_BUTTON_TEXT_SIZE / 2.,
                        egui::FontFamily::Name(PARAGRAPH_FONT_NAME.into()),
                    ),
                );

                if let Some(spectate_state) = STATES::get_spectate_state() {
                    if ui
                        .add_sized(
                            (window * PRIMARY_BUTTON_SIZE).cast(),
                            egui::Button::new(
                                egui::RichText::new(localization.translate("spectate"))
                                    .color(BUTTON_TEXT_COLOR),
                            )
                            .fill(BUTTON_COLOR),
                        )
                        .clicked()
                    {
                        game_state.set(spectate_state);
                        ctrl.pause_state.set(GamePauseState::Running);
                        ctrl.click_event.send(UiClickEvent);
                    }
                }

                ui.add_space(window.y * SPACE_SIZE);

                if ui
                    .add_sized(
                        (window * PRIMARY_BUTTON_SIZE).cast(),
                        egui::Button::new(
                            egui::RichText::new(localization.translate("exit-to-menu"))
                                .color(BUTTON_TEXT_COLOR),
                        )
                        .fill(BUTTON_COLOR),
                    )
                    .clicked()
                {
                    ctrl.app_state.set(AppState::AppMenu);
                    ctrl.click_event.send(UiClickEvent);
                }
            });
        });
}

fn keybinding_next_round<STATES: GameModeSubStates + RoundSystemSupport + FreelyMutableState>(
    settings: Res<ActiveSettingsBank>,
    input: Res<ButtonInput<KeyCode>>,
    mut next_state: ResMut<NextState<STATES>>,
    mut kb_event: EventWriter<KeyboardEvent>,
) {
    if settings
        .0
        .keybindings
        .next_round
        .any(|code| input.just_pressed(code))
    {
        next_state.set(STATES::get_inround_state());
        kb_event.send(KeyboardEvent);
    }
}
