pub mod chameleon_popup;
pub mod default_tile_popup;
pub mod market_popup;
pub mod rgb_popup;

use bevy::prelude::*;
use bevy_egui::egui::{self, load::SizedTexture};
use chameleon_popup::{chameleon_tile_clicked, render_chameleon_tile_popup};
use default_tile_popup::{default_tile_clicked, render_default_tile_popup};
use market_popup::{market_place_clicked, render_market_popup};
use rgb_popup::{render_rgb_tile_popup, rgb_tile_clicked};

use crate::{
    coordinates::CubeCoordinate,
    game::{
        gamemodes::{
            campaign::levels::{
                challenge::CampaignChallengeState, wettbewerb::CampaignWettbewerbState,
                CampaignGroups,
            },
            challenge::ChallengeGameState,
            creative::CreativeGameState,
            zen::ZenGameState,
            GameMode,
        },
        resources::marketplace::apply_changes_from_trade_ratio,
        tiles::{assets::TilePreviews, TileType},
        GamePauseState,
    },
    i18n::{Localization, Translate},
    prelude::WithTranslationID,
    settings::ActiveSettingsBank,
    ui::{
        commons::ShowGameUi,
        theme::{POPUP_COLOR_BG, POPUP_RADIUS, POPUP_STROKE_COLOR},
        WEBSITE_BASE,
    },
};

macro_rules! define_popup_ui_plugin {
    ($({
        // gamemode substate in which to interact with tiles
        state: $state:expr,
        // gamemode which is managed by this plugin
        gamemode: $gm:expr$(,)?
    }),*) => {
        pub struct TilePopupPlugin;
        impl Plugin for TilePopupPlugin {
            fn build(&self, app: &mut App) {
                use crate::prelude::never;

                app.add_systems(
                    Update,
                    (
                        // NOTE: if we ever use this plugin in a second gamemode
                        // one of these systems need a generic,
                        // so that bevy can differntiate between them
                        (
                            (default_tile_clicked, market_place_clicked, rgb_tile_clicked, chameleon_tile_clicked),
                            (
                                render_default_tile_popup,
                                render_market_popup.before(apply_changes_from_trade_ratio),
                                render_rgb_tile_popup,
                                render_chameleon_tile_popup,
                            ),
                        )
                            .chain()
                    )
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(in_state(GamePauseState::Running))
                        .run_if(in_state(ShowGameUi::On)),
                );
                app.add_systems(
                    Update,
                    keybinding_close_popup
                        .run_if(never()$(.or(in_state($state)))*)
                        .run_if(in_state(GamePauseState::Running)),
                );
                app.insert_resource(TileInfoWindow::default());
                $(
                    app.add_systems(OnEnter($gm), reset_tile_ui);
                )*
            }
        }
    }
}
define_popup_ui_plugin![
    {
        state: ChallengeGameState::InRound,
        gamemode: GameMode::Challenge
    },
    {
        state: CreativeGameState::InGame,
        gamemode: GameMode::Creative
    },
    {
        state: CampaignChallengeState::InRound,
        gamemode: CampaignGroups::CampaignChallenge
    },
    {
        state: CampaignWettbewerbState::InRound,
        gamemode: CampaignGroups::CampaignWettbewerb
    },
    {
        state: ZenGameState::InRound,
        gamemode: GameMode::Zen
    }
];

pub fn popup_window<R>(
    ctx: &mut egui::Context,
    open: &mut bool,
    name: &str,
    add_contents: impl FnOnce(&mut egui::Ui) -> R,
) {
    let my_frame = egui::Frame {
        fill: POPUP_COLOR_BG,
        stroke: egui::Stroke::new(1.5, POPUP_STROKE_COLOR),
        corner_radius: POPUP_RADIUS,
        inner_margin: egui::Margin::same(5),
        outer_margin: egui::Margin::same(0),
        ..Default::default()
    };
    egui::Window::new(name)
        .frame(my_frame)
        .collapsible(false)
        .resizable(true)
        .constrain_to(ctx.screen_rect())
        .max_height(ctx.screen_rect().height())
        .max_width(ctx.screen_rect().width())
        .movable(true)
        .default_pos(egui::Pos2 { x: 10.0, y: 100.0 })
        .open(open)
        .id("Tile-popup-window".into())
        .show(ctx, |ui| {
            // HACK: the .scroll window property caused the size
            // to be too big, this fits the content size better.
            // If the window gets to small, it can be made bigger,
            // by resizing it
            egui::ScrollArea::vertical().show(ui, add_contents)
        });
}

pub fn tile_info_display(
    ui: &mut egui::Ui,
    sel_tile: &TileType,
    sel_tile_coord: &CubeCoordinate,
    previews: &Res<TilePreviews>,
    localization: &Res<Localization>,
    settings: &Res<ActiveSettingsBank>,
) {
    // TODO: Make me pretty:
    // use icons + right align
    ui.horizontal(|ui| {
        if let Some(handle) = previews.get_egui_handle(*sel_tile) {
            ui.add(egui::widgets::Image::new(SizedTexture::new(
                handle,
                [28.0, 28.0],
            )));
        }
        ui.heading(localization.translate(sel_tile.get_translation_id()));
    });
    ui.label(localization.translate(&format!("{}-description", sel_tile.get_translation_id())));
    ui.hyperlink_to(
        localization.translate("tile-popup-more-information"),
        format!(
            "{}/{}/book/tiles/{}",
            WEBSITE_BASE,
            settings.lang.get_book_id(),
            sel_tile.get_book_link_url()
        ),
    );
    ui.label(format!(
        "{} {}×{}×{}",
        localization.translate("tile-popup-coordinates"),
        sel_tile_coord.q,
        sel_tile_coord.r,
        sel_tile_coord.s
    ));
}

#[derive(Resource, Default)]
pub struct TileInfoWindow {
    /// whether the tile-window is open
    /// needed so the Window can use .open() -> has button in top left to close it
    pub open: bool,
    /// Holds the CubeCoordinates of the tile currently selected by the player
    /// Needed for which tile should be displayed in the tile-window
    /// Should be None if the Tile-window is closed (open == false)
    pub selected_tile: Option<CubeCoordinate>,
}

fn keybinding_close_popup(
    settings: Res<ActiveSettingsBank>,
    input: Res<ButtonInput<KeyCode>>,
    mut tile_window: ResMut<TileInfoWindow>,
) {
    if settings
        .0
        .keybindings
        .close_popup
        .any(|code| input.just_pressed(code))
    {
        *tile_window = TileInfoWindow::default();
    }
}

fn reset_tile_ui(mut tile_window: ResMut<TileInfoWindow>) {
    *tile_window = TileInfoWindow::default();
}
